<?php
/**
 * Plugin Name: Marshal Mark - WP Config Analyzer
 * Plugin URI: https://kisscsabi.com/marshal-mark
 * Description: Analyzes and fixes common issues in your wp-config.php file.
 * Version: 1.1.9
 * Author: Jabybaby
 * Author URI: https://kisscsabi.com
 * Text Domain: marshal-mark
 * Update URI: https://warehouse.wp.dev3.smartbits.hu/plugins/marshal-mark
 */

// If this file is called directly, abort.
if (!defined('ABSPATH')) {
    exit;
}

// Define the main plugin file path constant
if (!defined(strtoupper(str_replace('-', '_', 'marshal-mark')) . '_FILE')) {
    define(strtoupper(str_replace('-', '_', 'marshal-mark')) . '_FILE', __FILE__);
}

// Include the auto-update functionality
require_once plugin_dir_path(__FILE__) . 'auto-updates.php';

// Initialize auto-update functionality
Marshal_Mark_Auto_Updates::init();

// Define plugin constants
define('MARSHAL_MARK_VERSION', '1.1.9');
define('MARSHAL_MARK_PLUGIN_DIR', plugin_dir_path(__FILE__));
define('MARSHAL_MARK_PLUGIN_URL', plugin_dir_url(__FILE__));

/**
 * The core plugin class
 */
class Marshal_Mark {

    /**
     * The unique instance of the plugin.
     *
     * @var Marshal_Mark
     */
    private static $instance;

    /**
     * Gets an instance of the plugin.
     *
     * @return Marshal_Mark
     */
    public static function get_instance() {
        if (null === self::$instance) {
            self::$instance = new self();
        }
        return self::$instance;
    }

    /**
     * Constructor.
     */
    private function __construct() {
        $this->load_dependencies();
        $this->define_admin_hooks();
    }

    /**
     * Load the required dependencies for this plugin.
     */
    private function load_dependencies() {
        // Include the config analyzer class
        require_once MARSHAL_MARK_PLUGIN_DIR . 'includes/class-marshal-mark-config-analyzer.php';
        
        // Include the admin class
        require_once MARSHAL_MARK_PLUGIN_DIR . 'admin/class-marshal-mark-admin.php';
    }

    /**
     * Register all of the hooks related to the admin area functionality.
     */
    private function define_admin_hooks() {
        $plugin_admin = new Marshal_Mark_Admin('marshal-mark', MARSHAL_MARK_VERSION);
        
        // Add admin menu
        add_action('admin_menu', array($plugin_admin, 'add_plugin_admin_menu'));
        
        // Add settings link to plugins page
        add_filter('plugin_action_links_' . plugin_basename(__FILE__), array($plugin_admin, 'add_action_links'));
        
        // Register admin styles and scripts
        add_action('admin_enqueue_scripts', array($plugin_admin, 'enqueue_styles'));
        add_action('admin_enqueue_scripts', array($plugin_admin, 'enqueue_scripts'));
        
        // Register settings
        add_action('admin_init', array($plugin_admin, 'register_settings'));
        
        // Register AJAX handlers
        add_action('wp_ajax_marshal_mark_analyze_config', array($plugin_admin, 'ajax_analyze_config'));
        add_action('wp_ajax_marshal_mark_fix_problem', array($plugin_admin, 'ajax_fix_problem'));
        add_action('wp_ajax_marshal_mark_fix_all_problems', array($plugin_admin, 'ajax_fix_all_problems'));
        add_action('wp_ajax_marshal_mark_fix_problems_by_severity', array($plugin_admin, 'ajax_fix_problems_by_severity'));
        add_action('wp_ajax_marshal_mark_view_backup', array($plugin_admin, 'ajax_view_backup'));
        add_action('wp_ajax_marshal_mark_restore_backup', array($plugin_admin, 'ajax_restore_backup'));
        
        // Display admin notices
        add_action('admin_notices', array($plugin_admin, 'display_admin_notices'));
    }

    /**
     * Activation hook.
     */
    public static function activate() {
        // Create backups directory
        $backups_dir = MARSHAL_MARK_PLUGIN_DIR . 'backups';
        if (!file_exists($backups_dir)) {
            wp_mkdir_p($backups_dir);
            
            // Add an index.php file to prevent directory listing
            file_put_contents($backups_dir . '/index.php', '<?php // Silence is golden');
        }
        
        // Create problems directory if it doesn't exist
        $problems_dir = MARSHAL_MARK_PLUGIN_DIR . 'includes/problems';
        if (!file_exists($problems_dir)) {
            wp_mkdir_p($problems_dir);
            
            // Add an index.php file to prevent directory listing
            file_put_contents($problems_dir . '/index.php', '<?php // Silence is golden');
        }
        
        // Initialize default settings
        if (!get_option('marshal_mark_settings')) {
            update_option('marshal_mark_settings', array(
                'auto_fix' => false,
                'config_backup' => true,
                'default_admin_username' => 'Smartbits',
                'debug_mode' => false
            ));
        }
    }

    /**
     * Deactivation hook.
     */
    public static function deactivate() {
        // Nothing to do here for now
    }
}

// Register activation and deactivation hooks
register_activation_hook(__FILE__, array('Marshal_Mark', 'activate'));
register_deactivation_hook(__FILE__, array('Marshal_Mark', 'deactivate'));

// Initialize the plugin
Marshal_Mark::get_instance(); 