<?php
/**
 * Problem: Post Revisions
 *
 * Checks if WP_POST_REVISIONS is limited.
 *
 * @package Marshal_Mark
 * @since 1.0.0
 */

// If this file is called directly, abort.
if (!defined('ABSPATH')) {
    exit;
}

return array(
    'id' => 'post_revisions',
    'title' => __('Unlimited Post Revisions', 'marshal-mark'),
    'description' => __('WP_POST_REVISIONS is not limited. This can lead to database bloat.', 'marshal-mark'),
    'severity' => 'medium',
    'detection' => function($config_content) {
        return !preg_match("/define\s*\(\s*['\"]WP_POST_REVISIONS['\"]\s*,\s*(\d+|false)\s*\)\s*;/i", $config_content);
    },
    'fix' => array(
        'title' => __('Limit Post Revisions', 'marshal-mark'),
        'description' => __('Set WP_POST_REVISIONS to 5 in wp-config.php.', 'marshal-mark'),
        'code' => "define('WP_POST_REVISIONS', 5);",
        'implementation' => function($config_content) {
            // Check if WP_POST_REVISIONS is already defined
            if (preg_match("/define\s*\(\s*['\"]WP_POST_REVISIONS['\"]\s*,/i", $config_content)) {
                // Replace existing definition
                $pattern = "/define\s*\(\s*['\"]WP_POST_REVISIONS['\"]\s*,\s*.*?\s*\)\s*;/i";
                $replacement = "define('WP_POST_REVISIONS', 5); /* Marshal Mark: Changed to limit to 5 */";
                
                $new_content = preg_replace($pattern, $replacement, $config_content);
            } else {
                // Add new definition before "That's all"
                $pattern = "/(\/\* That's all, stop editing! Happy publishing\. \*\/)/i";
                $replacement = "define('WP_POST_REVISIONS', 5); /* Marshal Mark: Added */\n\n$1";
                
                $new_content = preg_replace($pattern, $replacement, $config_content);
            }
            
            return $new_content;
        }
    ),
    'stats' => function() {
        global $wpdb;
        
        // Get total number of revisions
        $total_revisions = $wpdb->get_var("SELECT COUNT(*) FROM $wpdb->posts WHERE post_type = 'revision'");
        
        // Get number of posts with more than 5 revisions
        $posts_with_excess_revisions = $wpdb->get_var("
            SELECT COUNT(DISTINCT post_parent) 
            FROM $wpdb->posts 
            WHERE post_type = 'revision' 
            GROUP BY post_parent 
            HAVING COUNT(*) > 5
        ");
        
        // Get number of excess revisions (revisions beyond 5 per post)
        $excess_revisions = $wpdb->get_var("
            SELECT SUM(revision_count - 5) 
            FROM (
                SELECT post_parent, COUNT(*) as revision_count 
                FROM $wpdb->posts 
                WHERE post_type = 'revision' 
                GROUP BY post_parent 
                HAVING COUNT(*) > 5
            ) as subquery
        ");
        
        return array(
            'total_revisions' => (int) $total_revisions,
            'posts_with_excess_revisions' => (int) $posts_with_excess_revisions,
            'excess_revisions' => (int) $excess_revisions
        );
    }
); 