<?php

/**
 * The config analyzer functionality of the plugin.
 *
 * @since      1.0.0
 */
class Marshal_Mark_Config_Analyzer {

    /**
     * The path to wp-config.php.
     *
     * @since    1.0.0
     * @access   private
     * @var      string    $config_path    The path to wp-config.php.
     */
    private $config_path;

    /**
     * The content of wp-config.php.
     *
     * @since    1.0.0
     * @access   private
     * @var      string    $config_content    The content of wp-config.php.
     */
    private $config_content;

    /**
     * The problem definitions.
     *
     * @since    1.0.0
     * @access   private
     * @var      array    $problems    The problem definitions.
     */
    private $problems = array();

    /**
     * Initialize the class and set its properties.
     *
     * @since    1.0.0
     */
    public function __construct() {
        $this->config_path = ABSPATH . 'wp-config.php';
        $this->refresh_config_content();
        $this->load_problem_definitions();
    }

    /**
     * Refresh the config content.
     *
     * @since    1.0.0
     */
    private function refresh_config_content() {
        if (file_exists($this->config_path)) {
            $this->config_content = file_get_contents($this->config_path);
        } else {
            $this->config_content = '';
        }
    }

    /**
     * Load problem definitions from files.
     *
     * @since    1.0.0
     */
    private function load_problem_definitions() {
        $problems_dir = MARSHAL_MARK_PLUGIN_DIR . 'includes/problems';
        $debug_info = array(
            'problems_dir_exists' => file_exists($problems_dir),
            'problems_dir_path' => $problems_dir,
            'files_found' => array(),
            'problems_loaded' => array()
        );
        
        if (!file_exists($problems_dir)) {
            // Store debug info in a transient
            set_transient('marshal_mark_debug_info', $debug_info, HOUR_IN_SECONDS);
            return;
        }
        
        $files = scandir($problems_dir);
        $debug_info['files_in_dir'] = $files;
        
        foreach ($files as $file) {
            if ($file === '.' || $file === '..' || pathinfo($file, PATHINFO_EXTENSION) !== 'php' || $file === 'index.php') {
                continue;
            }
            
            $debug_info['files_found'][] = $file;
            
            try {
                $problem = include $problems_dir . '/' . $file;
                
                if (is_array($problem) && isset($problem['id'])) {
                    // Add stats if there's a stats callback
                    if (isset($problem['stats']) && is_callable($problem['stats'])) {
                        $problem['stats'] = call_user_func($problem['stats']);
                    }
                    
                    $this->problems[$problem['id']] = $problem;
                    $debug_info['problems_loaded'][] = $problem['id'];
                } else {
                    $debug_info['load_errors'][] = "File $file did not return a valid problem definition";
                }
            } catch (Exception $e) {
                $debug_info['load_errors'][] = "Error loading $file: " . $e->getMessage();
            }
        }
        
        // Store debug info in a transient
        set_transient('marshal_mark_debug_info', $debug_info, HOUR_IN_SECONDS);
    }

    /**
     * Analyze the wp-config.php file for common issues.
     *
     * @since    1.0.0
     * @return   array    The analysis results.
     */
    public function analyze_config() {
        // Refresh the config content
        $this->refresh_config_content();
        
        // Initialize results
        $results = array(
            'is_writable' => is_writable($this->config_path),
            'problems' => array(),
            'problems_by_severity' => array(
                'critical' => array(),
                'high' => array(),
                'medium' => array(),
                'low' => array()
            )
        );
        
        // Check for problems using the loaded definitions
        foreach ($this->problems as $id => $problem) {
            if (isset($problem['detection']) && is_callable($problem['detection'])) {
                $has_problem = call_user_func($problem['detection'], $this->config_content);
                
                if ($has_problem) {
                    $results['problems'][$id] = $problem;
                    
                    // Group by severity
                    $severity = isset($problem['severity']) ? $problem['severity'] : 'medium';
                    if (!isset($results['problems_by_severity'][$severity])) {
                        $results['problems_by_severity'][$severity] = array();
                    }
                    $results['problems_by_severity'][$severity][$id] = $problem;
                }
            }
        }
        
        return $results;
    }

    /**
     * Fix a specific problem.
     *
     * @since    1.0.0
     * @param    string    $problem_id    The problem ID.
     * @return   boolean                  True if fixed, false otherwise.
     */
    public function fix_problem($problem_id) {
        // Refresh the config content
        $this->refresh_config_content();
        
        // Check if the problem exists
        if (!isset($this->problems[$problem_id])) {
            return false;
        }
        
        $problem = $this->problems[$problem_id];
        
        // Check if the problem has a fix implementation
        if (!isset($problem['fix']['implementation']) || !is_callable($problem['fix']['implementation'])) {
            return false;
        }
        
        // Create a backup before making changes
        $this->create_backup();
        
        // Apply the fix
        $new_content = call_user_func($problem['fix']['implementation'], $this->config_content);
        
        if ($new_content !== $this->config_content) {
            // Write the changes to the file
            if (file_put_contents($this->config_path, $new_content)) {
                $this->refresh_config_content();
                return true;
            }
        }
        
        return false;
    }

    /**
     * Fix problems with a specific severity level or higher.
     *
     * @since    1.0.0
     * @param    string    $severity    The severity level ('critical', 'high', 'medium', 'low').
     * @return   array                  The results of the fix operation.
     */
    public function fix_problems_by_severity($severity = 'high') {
        // Analyze the config to get all problems
        $analysis = $this->analyze_config();
        
        // Initialize results
        $results = array(
            'success' => array(),
            'failure' => array()
        );
        
        // Create a backup before making changes
        $this->create_backup();
        
        // Define severity levels and their order
        $severity_levels = array(
            'critical' => 4,
            'high' => 3,
            'medium' => 2,
            'low' => 1
        );
        
        // Get the minimum severity level to fix
        $min_severity = isset($severity_levels[$severity]) ? $severity_levels[$severity] : $severity_levels['high'];
        
        // Fix problems with the specified severity or higher
        foreach ($analysis['problems'] as $id => $problem) {
            $problem_severity = isset($problem['severity']) ? $problem['severity'] : 'medium';
            $problem_severity_level = isset($severity_levels[$problem_severity]) ? $severity_levels[$problem_severity] : $severity_levels['medium'];
            
            // Only fix problems with the specified severity or higher
            if ($problem_severity_level >= $min_severity) {
                if (isset($problem['fix']['implementation']) && is_callable($problem['fix']['implementation'])) {
                    $fixed = $this->fix_problem($id);
                    
                    if ($fixed) {
                        $results['success'][] = $id;
                    } else {
                        $results['failure'][] = $id;
                    }
                }
            }
        }
        
        return $results;
    }

    /**
     * Create a backup of wp-config.php.
     *
     * @since    1.0.0
     * @return   boolean    True if backup created, false otherwise.
     */
    private function create_backup() {
        $backups_dir = MARSHAL_MARK_PLUGIN_DIR . 'backups';
        
        // Ensure the backups directory exists
        if (!file_exists($backups_dir)) {
            wp_mkdir_p($backups_dir);
        }
        
        // Create a backup with timestamp
        $backup_path = $backups_dir . '/wp-config-backup-' . date('Y-m-d-H-i-s') . '.php';
        
        return copy($this->config_path, $backup_path);
    }
}