<?php

/**
 * Provide a admin area view for the plugin's settings page
 *
 * This file is used to markup the admin-facing aspects of the plugin.
 *
 * @since      1.0.0
 */
?>

<div class="wrap">
    <div class="marshal-mark-admin-header">
        <div class="marshal-mark-logo">
            <img src="<?php echo plugin_dir_url(dirname(__FILE__)) . 'images/logo.webp'; ?>" alt="Marshal Mark Logo">
        </div>
        <div class="marshal-mark-intro">
            <h1><?php _e('Settings', 'marshal-mark'); ?></h1>
            <p><?php _e('Configure how Marshal Mark operates on your WordPress site.', 'marshal-mark'); ?></p>
        </div>
    </div>

    <div class="marshal-mark-admin-content">
        <form method="post" action="options.php">
            <?php
                // Output security fields
                settings_fields('marshal_mark_options');
                
                // Output setting sections
                do_settings_sections('marshal_mark_settings');
                
                // Submit button
                submit_button();
            ?>
        </form>
        
        <div class="marshal-mark-backups">
            <h3><?php _e('Configuration Backups', 'marshal-mark'); ?></h3>
            
            <?php
            $backups_dir = MARSHAL_MARK_PLUGIN_DIR . 'backups/';
            $backups = array();
            
            if (file_exists($backups_dir)) {
                $files = scandir($backups_dir);
                
                foreach ($files as $file) {
                    if ($file != '.' && $file != '..' && strpos($file, 'wp-config-backup-') === 0 && pathinfo($file, PATHINFO_EXTENSION) === 'php') {
                        $backups[] = $file;
                    }
                }
                
                // Sort backups by date (newest first)
                usort($backups, function($a, $b) use ($backups_dir) {
                    return filemtime($backups_dir . $b) - filemtime($backups_dir . $a);
                });
            }
            ?>
            
            <?php if (empty($backups)): ?>
                <p><?php _e('No backups found.', 'marshal-mark'); ?></p>
            <?php else: ?>
                <p><?php _e('The following backups of your wp-config.php file are available:', 'marshal-mark'); ?></p>
                
                <table class="widefat">
                    <thead>
                        <tr>
                            <th><?php _e('Backup Date', 'marshal-mark'); ?></th>
                            <th><?php _e('File Size', 'marshal-mark'); ?></th>
                            <th><?php _e('Actions', 'marshal-mark'); ?></th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php foreach ($backups as $backup): ?>
                            <tr>
                                <td>
                                    <?php
                                    $backup_time = filemtime($backups_dir . $backup);
                                    echo date_i18n(get_option('date_format') . ' ' . get_option('time_format'), $backup_time);
                                    ?>
                                </td>
                                <td>
                                    <?php
                                    $size = filesize($backups_dir . $backup);
                                    echo size_format($size);
                                    ?>
                                </td>
                                <td>
                                    <a href="#" class="button view-backup" data-backup="<?php echo esc_attr($backup); ?>"><?php _e('View', 'marshal-mark'); ?></a>
                                    <a href="#" class="button restore-backup" data-backup="<?php echo esc_attr($backup); ?>"><?php _e('Restore', 'marshal-mark'); ?></a>
                                </td>
                            </tr>
                        <?php endforeach; ?>
                    </tbody>
                </table>
            <?php endif; ?>
        </div>
    </div>
</div>

<!-- Backup Viewer Modal -->
<div id="backup-viewer-modal" style="display: none;">
    <div class="backup-viewer-content">
        <h3 id="backup-title"><?php _e('Backup Content', 'marshal-mark'); ?></h3>
        <pre id="backup-content"></pre>
        <div class="backup-viewer-actions">
            <button class="button close-modal"><?php _e('Close', 'marshal-mark'); ?></button>
        </div>
    </div>
</div>

<script>
jQuery(document).ready(function($) {
    // View backup
    $('.view-backup').on('click', function(e) {
        e.preventDefault();
        
        var backup = $(this).data('backup');
        var $modal = $('#backup-viewer-modal');
        var $content = $('#backup-content');
        var $title = $('#backup-title');
        
        // Show loading state
        $content.html('<?php _e('Loading backup content...', 'marshal-mark'); ?>');
        $title.text('<?php _e('Backup Content', 'marshal-mark'); ?>: ' + backup);
        $modal.show();
        
        // Fetch backup content via AJAX
        $.ajax({
            url: marshal_mark_data.ajax_url,
            type: 'POST',
            data: {
                action: 'marshal_mark_view_backup',
                nonce: marshal_mark_data.nonce,
                backup: backup
            },
            success: function(response) {
                if (response.success) {
                    $content.html(response.data.content);
                } else {
                    $content.html('<?php _e('Error loading backup content:', 'marshal-mark'); ?> ' + response.data.message);
                }
            },
            error: function() {
                $content.html('<?php _e('Error loading backup content. Please try again.', 'marshal-mark'); ?>');
            }
        });
    });
    
    // Close modal
    $('.close-modal').on('click', function() {
        $('#backup-viewer-modal').hide();
    });
    
    // Close modal when clicking outside the content
    $('#backup-viewer-modal').on('click', function(e) {
        if (e.target === this) {
            $(this).hide();
        }
    });
    
    // Restore backup
    $('.restore-backup').on('click', function(e) {
        e.preventDefault();
        
        var backup = $(this).data('backup');
        
        if (confirm('<?php _e('Are you sure you want to restore this backup? This will overwrite your current wp-config.php file.', 'marshal-mark'); ?>')) {
            // Show loading state
            var $button = $(this);
            var originalText = $button.text();
            $button.prop('disabled', true).text('<?php _e('Restoring...', 'marshal-mark'); ?>');
            
            // Restore backup via AJAX
            $.ajax({
                url: marshal_mark_data.ajax_url,
                type: 'POST',
                data: {
                    action: 'marshal_mark_restore_backup',
                    nonce: marshal_mark_data.nonce,
                    backup: backup
                },
                success: function(response) {
                    $button.prop('disabled', false).text(originalText);
                    
                    if (response.success) {
                        alert('<?php _e('Backup restored successfully!', 'marshal-mark'); ?>');
                        location.reload();
                    } else {
                        alert('<?php _e('Error restoring backup:', 'marshal-mark'); ?> ' + response.data.message);
                    }
                },
                error: function() {
                    $button.prop('disabled', false).text(originalText);
                    alert('<?php _e('Error restoring backup. Please try again.', 'marshal-mark'); ?>');
                }
            });
        }
    });
});
</script> 