<?php

/**
 * Provide a admin area view for the plugin
 *
 * This file is used to markup the admin-facing aspects of the plugin.
 *
 * @since      1.0.0
 */

// Get the config analyzer
$config_analyzer = new Marshal_Mark_Config_Analyzer();

// Get the analysis results
$analysis = $config_analyzer->analyze_config();
?>

<div class="wrap">
    <div class="marshal-mark-admin-header">
        <div class="marshal-mark-logo">
            <img src="<?php echo plugin_dir_url(dirname(__FILE__)) . 'images/logo.webp'; ?>" alt="Marshal Mark Logo">
        </div>
        <div class="marshal-mark-intro">
            <h1>Configuration Analyzer</h1>
            <p>Marshal Mark analyzes your WordPress configuration file (wp-config.php) for potential issues and helps you fix them.</p>
        </div>
    </div>

    <div class="marshal-mark-admin-content">
        <div class="marshal-mark-config-status">
            <h3><?php _e('WordPress Configuration Status', 'marshal-mark'); ?></h3>
            
            <div class="config-info">
                <p><strong><?php _e('wp-config.php Status:', 'marshal-mark'); ?></strong> 
                    <?php if ($analysis['is_writable']): ?>
                        <span class="status-good"><?php _e('Writable', 'marshal-mark'); ?></span>
                    <?php else: ?>
                        <span class="status-bad"><?php _e('Not Writable', 'marshal-mark'); ?></span>
                        <div class="notice notice-warning">
                            <p><?php _e('Your wp-config.php file is not writable. Marshal Mark will not be able to automatically fix issues.', 'marshal-mark'); ?></p>
                        </div>
                    <?php endif; ?>
                </p>
            </div>
        </div>
        
        <div class="marshal-mark-analysis">
            <div class="analysis-header">
                <h3><?php _e('Configuration Analysis', 'marshal-mark'); ?></h3>
                <div class="analysis-actions">
                    <button id="marshal-mark-analyze" class="button button-primary"><?php _e('Analyze Now', 'marshal-mark'); ?></button>
                    <?php if (!empty($analysis['problems'])): ?>
                        <button id="marshal-mark-fix-all" class="button"><?php _e('Fix All Issues', 'marshal-mark'); ?></button>
                    <?php endif; ?>
                </div>
            </div>
            
            <div id="marshal-mark-analysis-results">
                <?php if (empty($analysis['problems'])): ?>
                    <div class="notice notice-success">
                        <p><?php _e('No issues found in your WordPress configuration. Great job!', 'marshal-mark'); ?></p>
                    </div>
                <?php else: ?>
                    <div class="marshal-mark-severity-actions-clean">
                        <p><?php _e('Fix issues by severity:', 'marshal-mark'); ?></p>
                        <div class="severity-buttons">
                            <?php if (!empty($analysis['problems_by_severity']['critical'])): ?>
                                <button class="button fix-by-severity" data-severity="critical">
                                    <?php _e('Fix Critical Issues', 'marshal-mark'); ?> 
                                    <span class="count">(<?php echo count($analysis['problems_by_severity']['critical']); ?>)</span>
                                </button>
                            <?php endif; ?>
                            
                            <?php if (!empty($analysis['problems_by_severity']['high'])): ?>
                                <button class="button fix-by-severity" data-severity="high">
                                    <?php _e('Fix High Severity Issues', 'marshal-mark'); ?> 
                                    <span class="count">(<?php echo count($analysis['problems_by_severity']['high']); ?>)</span>
                                </button>
                            <?php endif; ?>
                            
                            <?php if (!empty($analysis['problems_by_severity']['medium'])): ?>
                                <button class="button fix-by-severity" data-severity="medium">
                                    <?php _e('Fix Medium Severity Issues', 'marshal-mark'); ?> 
                                    <span class="count">(<?php echo count($analysis['problems_by_severity']['medium']); ?>)</span>
                                </button>
                            <?php endif; ?>
                            
                            <?php if (!empty($analysis['problems_by_severity']['low'])): ?>
                                <button class="button fix-by-severity" data-severity="low">
                                    <?php _e('Fix Low Severity Issues', 'marshal-mark'); ?> 
                                    <span class="count">(<?php echo count($analysis['problems_by_severity']['low']); ?>)</span>
                                </button>
                            <?php endif; ?>
                        </div>
                    </div>
                    
                    <?php 
                    // Define severity order and labels
                    $severities = array(
                        'critical' => __('Critical Issues', 'marshal-mark'),
                        'high' => __('High Severity Issues', 'marshal-mark'),
                        'medium' => __('Medium Severity Issues', 'marshal-mark'),
                        'low' => __('Low Severity Issues', 'marshal-mark')
                    );
                    
                    // Display problems grouped by severity
                    foreach ($severities as $severity => $label):
                        if (!empty($analysis['problems_by_severity'][$severity])):
                    ?>
                        <div class="marshal-mark-severity-group severity-<?php echo esc_attr($severity); ?>">
                            <h4><?php echo esc_html($label); ?> (<?php echo count($analysis['problems_by_severity'][$severity]); ?>)</h4>
                            
                            <table class="widefat marshal-mark-problems-table">
                                <thead>
                                    <tr>
                                        <th><?php _e('Issue', 'marshal-mark'); ?></th>
                                        <th><?php _e('Description', 'marshal-mark'); ?></th>
                                        <th><?php _e('Fix', 'marshal-mark'); ?></th>
                                        <th><?php _e('Actions', 'marshal-mark'); ?></th>
                                    </tr>
                                </thead>
                                <tbody>
                                    <?php foreach ($analysis['problems_by_severity'][$severity] as $id => $problem): ?>
                                        <tr class="problem-row" data-problem-id="<?php echo esc_attr($id); ?>">
                                            <td class="problem-title"><?php echo esc_html($problem['title']); ?></td>
                                            <td class="problem-description">
                                                <?php echo esc_html($problem['description']); ?>
                                                
                                                <?php if ($id === 'post_revisions' && isset($problem['stats'])): ?>
                                                    <div class="problem-stats">
                                                        <p><strong><?php _e('Revision Statistics:', 'marshal-mark'); ?></strong></p>
                                                        <ul>
                                                            <li><?php printf(__('Total revisions: %d', 'marshal-mark'), $problem['stats']['total_revisions']); ?></li>
                                                            <li><?php printf(__('Posts with excess revisions: %d', 'marshal-mark'), $problem['stats']['posts_with_excess_revisions']); ?></li>
                                                            <li><?php printf(__('Excess revisions that will be deleted: %d', 'marshal-mark'), $problem['stats']['excess_revisions']); ?></li>
                                                        </ul>
                                                    </div>
                                                <?php endif; ?>
                                                
                                                <?php if ($id === 'comments_enabled' && isset($problem['stats'])): ?>
                                                    <div class="problem-stats">
                                                        <p><strong><?php _e('Comments Statistics:', 'marshal-mark'); ?></strong></p>
                                                        <ul>
                                                            <li><?php printf(__('Default comment status: %s', 'marshal-mark'), $problem['stats']['default_status'] === 'open' ? __('Open', 'marshal-mark') : __('Closed', 'marshal-mark')); ?></li>
                                                            <li><?php printf(__('Posts with open comments: %d', 'marshal-mark'), $problem['stats']['open_posts_count']); ?></li>
                                                            <li><?php printf(__('Pages with open comments: %d', 'marshal-mark'), $problem['stats']['open_pages_count']); ?></li>
                                                        </ul>
                                                    </div>
                                                <?php endif; ?>
                                                
                                                <?php if ($id === 'default_admin_user_exists' && isset($problem['stats']) && $problem['stats']['exists']): ?>
                                                    <div class="problem-stats">
                                                        <p><strong><?php _e('User Information:', 'marshal-mark'); ?></strong></p>
                                                        <ul>
                                                            <li><?php printf(__('Username: %s', 'marshal-mark'), esc_html($problem['stats']['username'])); ?></li>
                                                            <li><?php printf(__('User ID: %d', 'marshal-mark'), $problem['stats']['user_id']); ?></li>
                                                            <li><?php printf(__('Email: %s', 'marshal-mark'), esc_html($problem['stats']['email'])); ?></li>
                                                            <li><?php printf(__('Is Administrator: %s', 'marshal-mark'), $problem['stats']['is_admin'] ? __('Yes', 'marshal-mark') : __('No', 'marshal-mark')); ?></li>
                                                            <li><?php printf(__('Created: %s', 'marshal-mark'), date_i18n(get_option('date_format') . ' ' . get_option('time_format'), strtotime($problem['stats']['created']))); ?></li>
                                                        </ul>
                                                    </div>
                                                <?php endif; ?>
                                                
                                                <?php if ($id === 'default_admin_user_missing' && isset($problem['stats'])): ?>
                                                    <div class="problem-stats">
                                                        <p><strong><?php _e('User Information:', 'marshal-mark'); ?></strong></p>
                                                        <ul>
                                                            <li><?php printf(__('Default username: %s', 'marshal-mark'), esc_html($problem['stats']['username'])); ?></li>
                                                            <li><?php _e('Status: Not created yet', 'marshal-mark'); ?></li>
                                                        </ul>
                                                    </div>
                                                <?php endif; ?>
                                                
                                                <?php if ($id === 'visual_regression_detected' && isset($problem['stats'])): ?>
                                                    <div class="problem-stats">
                                                        <p><strong><?php _e('Visual Regression Information:', 'marshal-mark'); ?></strong></p>
                                                        <ul>
                                                            <li><?php printf(__('Last check: %s', 'marshal-mark'), $problem['stats']['last_check']); ?></li>
                                                            <li><?php printf(__('Changes detected: %s', 'marshal-mark'), $problem['stats']['changes_detected'] ? __('Yes', 'marshal-mark') : __('No', 'marshal-mark')); ?></li>
                                                        </ul>
                                                        
                                                        <?php if (!empty($problem['stats']['changed_pages'])): ?>
                                                            <p><strong><?php _e('Changed Pages:', 'marshal-mark'); ?></strong></p>
                                                            <ul class="changed-pages-list">
                                                                <?php foreach ($problem['stats']['changed_pages'] as $page): ?>
                                                                    <li>
                                                                        <a href="<?php echo esc_url($page['url']); ?>" target="_blank"><?php echo esc_html($page['title']); ?></a>
                                                                        <div class="visual-diff-images">
                                                                            <div class="visual-diff-image">
                                                                                <p><?php _e('Before:', 'marshal-mark'); ?></p>
                                                                                <img src="<?php echo esc_url($page['before_image']); ?>" alt="<?php echo esc_attr(sprintf(__('Before image of %s', 'marshal-mark'), $page['title'])); ?>" width="300">
                                                                            </div>
                                                                            <div class="visual-diff-image">
                                                                                <p><?php _e('After:', 'marshal-mark'); ?></p>
                                                                                <img src="<?php echo esc_url($page['after_image']); ?>" alt="<?php echo esc_attr(sprintf(__('After image of %s', 'marshal-mark'), $page['title'])); ?>" width="300">
                                                                            </div>
                                                                        </div>
                                                                    </li>
                                                                <?php endforeach; ?>
                                                            </ul>
                                                        <?php endif; ?>
                                                    </div>
                                                <?php endif; ?>
                                            </td>
                                            <td class="problem-fix">
                                                <strong><?php echo esc_html($problem['fix']['title']); ?></strong>
                                                <p><?php echo esc_html($problem['fix']['description']); ?></p>
                                                <pre><code><?php echo esc_html($problem['fix']['code']); ?></code></pre>
                                            </td>
                                            <td class="problem-actions">
                                                <button class="button fix-problem" data-problem-id="<?php echo esc_attr($id); ?>"><?php _e('Fix Now', 'marshal-mark'); ?></button>
                                            </td>
                                        </tr>
                                    <?php endforeach; ?>
                                </tbody>
                            </table>
                        </div>
                    <?php 
                        endif;
                    endforeach; 
                    ?>
                <?php endif; ?>
            </div>
        </div>
        
        <?php 
        // Display debug info if the method exists and debug mode is enabled
        if (method_exists($this, 'display_debug_info')) {
            $this->display_debug_info();
        }
        ?>
    </div>
</div>

<script>
jQuery(document).ready(function($) {
    // Analyze button
    $('#marshal-mark-analyze').on('click', function() {
        var $button = $(this);
        var originalText = $button.text();
        
        // Disable button and show loading state
        $button.prop('disabled', true).text('<?php _e('Analyzing...', 'marshal-mark'); ?>');
        
        // Make AJAX request
        $.ajax({
            url: marshal_mark_data.ajax_url,
            type: 'POST',
            data: {
                action: 'marshal_mark_analyze_config',
                nonce: marshal_mark_data.nonce
            },
            success: function(response) {
                // Re-enable button
                $button.prop('disabled', false).text(originalText);
                
                if (response.success) {
                    // Reload the page to show updated results
                    location.reload();
                } else {
                    alert(response.data.message || '<?php _e('An error occurred while analyzing the configuration.', 'marshal-mark'); ?>');
                }
            },
            error: function() {
                // Re-enable button
                $button.prop('disabled', false).text(originalText);
                
                alert('<?php _e('An error occurred while analyzing the configuration.', 'marshal-mark'); ?>');
            }
        });
    });
    
    // Fix problem button
    $('.fix-problem').on('click', function() {
        var $button = $(this);
        var problemId = $button.data('problem-id');
        var originalText = $button.text();
        
        // Disable button and show loading state
        $button.prop('disabled', true).text('<?php _e('Fixing...', 'marshal-mark'); ?>');
        
        // Make AJAX request
        $.ajax({
            url: marshal_mark_data.ajax_url,
            type: 'POST',
            data: {
                action: 'marshal_mark_fix_problem',
                nonce: marshal_mark_data.nonce,
                problem_id: problemId
            },
            success: function(response) {
                // Re-enable button
                $button.prop('disabled', false).text(originalText);
                
                if (response.success) {
                    // Reload the page to show updated results
                    location.reload();
                } else {
                    alert(response.data.message || '<?php _e('An error occurred while fixing the problem.', 'marshal-mark'); ?>');
                }
            },
            error: function() {
                // Re-enable button
                $button.prop('disabled', false).text(originalText);
                
                alert('<?php _e('An error occurred while fixing the problem.', 'marshal-mark'); ?>');
            }
        });
    });
    
    // Fix all button
    $('#marshal-mark-fix-all').on('click', function() {
        var $button = $(this);
        var originalText = $button.text();
        
        // Confirm before fixing all
        if (!confirm('<?php _e('Are you sure you want to fix all issues? This will modify your wp-config.php file.', 'marshal-mark'); ?>')) {
            return;
        }
        
        // Disable button and show loading state
        $button.prop('disabled', true).text('<?php _e('Fixing...', 'marshal-mark'); ?>');
        
        // Make AJAX request
        $.ajax({
            url: marshal_mark_data.ajax_url,
            type: 'POST',
            data: {
                action: 'marshal_mark_fix_all_problems',
                nonce: marshal_mark_data.nonce
            },
            success: function(response) {
                // Re-enable button
                $button.prop('disabled', false).text(originalText);
                
                if (response.success) {
                    // Reload the page to show updated results
                    location.reload();
                } else {
                    alert(response.data.message || '<?php _e('An error occurred while fixing the problems.', 'marshal-mark'); ?>');
                }
            },
            error: function() {
                // Re-enable button
                $button.prop('disabled', false).text(originalText);
                
                alert('<?php _e('An error occurred while fixing the problems.', 'marshal-mark'); ?>');
            }
        });
    });
    
    // Fix by severity button
    $('.fix-by-severity').on('click', function() {
        var $button = $(this);
        var severity = $button.data('severity');
        var originalText = $button.text();
        
        // Confirm before fixing
        if (!confirm('<?php _e('Are you sure you want to fix all issues with this severity level? This will modify your wp-config.php file.', 'marshal-mark'); ?>')) {
            return;
        }
        
        // Disable button and show loading state
        $button.prop('disabled', true).text('<?php _e('Fixing...', 'marshal-mark'); ?>');
        
        // Make AJAX request
        $.ajax({
            url: marshal_mark_data.ajax_url,
            type: 'POST',
            data: {
                action: 'marshal_mark_fix_problems_by_severity',
                nonce: marshal_mark_data.nonce,
                severity: severity
            },
            success: function(response) {
                // Re-enable button
                $button.prop('disabled', false).text(originalText);
                
                if (response.success) {
                    // Reload the page to show updated results
                    location.reload();
                } else {
                    alert(response.data.message || '<?php _e('An error occurred while fixing the problems.', 'marshal-mark'); ?>');
                }
            },
            error: function() {
                // Re-enable button
                $button.prop('disabled', false).text(originalText);
                
                alert('<?php _e('An error occurred while fixing the problems.', 'marshal-mark'); ?>');
            }
        });
    });
});
</script>