<?php
/*
* Plugin Name: HelloPack Manager
* Description: Exposes endpoints to manage HelloPack client
* Version: 1.1.4
* Author: Jabybaby
* Update URI: https://warehouse.wp.dev3.smartbits.hu/plugins/hellopack-management
*/

// Prevent direct access to this file
if (!defined('ABSPATH')) {
    exit;
}

// Define the main plugin file path constant
if (!defined('HELLOPACK_MANAGER_FILE')) {
    define('HELLOPACK_MANAGER_FILE', __FILE__);
}

// Include the auto-update functionality
require_once plugin_dir_path(__FILE__) . 'auto-updates.php';

// Initialize auto-update functionality
HelloPack_Auto_Updates::init();

// Register the REST API endpoint
add_action('rest_api_init', function () {
    register_rest_route('hellopack/v1', '/set-token', array(
        'methods' => 'POST',
        'callback' => 'handle_set_token',
        'permission_callback' => 'check_admin_permission',
        'args' => array(
            'license_key' => array(
                'required' => true,
                'type' => 'string',
            ),
        ),
    ));
});

// Check if user has admin permissions
function check_admin_permission() {
    return current_user_can('manage_options');
}

// Handle the token setting
function handle_set_token($request) {
    $license_key = sanitize_text_field($request['license_key']);
    
    $result = update_option('hellopack_client', array(
        'token' => $license_key,
    ));
    
    if ($result) {
        return new WP_REST_Response(array(
            'success' => true,
            'message' => 'HelloPack client token updated successfully'
        ), 200);
    }
    
    return new WP_REST_Response(array(
        'success' => false,
        'message' => 'Failed to update HelloPack client token'
    ), 400);
} 

class HelloPack_Manager {
    
    public function __construct() {
        add_action('rest_api_init', array($this, 'register_endpoints'));
    }

    /**
     * Register the REST API endpoints
     */
    public function register_endpoints() {
        register_rest_route('hellopack/v1', '/install/(?P<plugin_id>\d+)', array(
            'methods' => 'GET',
            'callback' => array($this, 'handle_installation'),
            'permission_callback' => '__return_true', // Allow public access
            'args' => array(
                'plugin_id' => array(
                    'required' => true,
                    'type' => 'integer',
                )
            )
        ));

        // Health check endpoint
        register_rest_route('hellopack/v1', '/health', array(
            'methods' => 'GET',
            'callback' => array($this, 'handle_health_check'),
            'permission_callback' => '__return_true' // Allow public access
        ));

        // New search endpoint
        register_rest_route('hellopack/v1', '/search', array(
            'methods' => 'GET',
            'callback' => array($this, 'handle_search'),
            'permission_callback' => '__return_true',
            'args' => array(
                'query' => array(
                    'required' => true,
                    'type' => 'string',
                )
            )
        ));
    }

    /**
     * Handle health check request
     * Returns status of HelloPack client configuration
     */
    public function handle_health_check() {
        $hellopack_option = get_option('hellopack_client');
        $has_token = !empty($hellopack_option) && !empty($hellopack_option['token']);
        
        return new WP_REST_Response(array(
            'success' => true,
            'status' => $has_token ? 'configured' : 'not_configured',
            'has_token' => $has_token
        ));
    }

    /**
     * Handle the plugin installation
     */
    public function handle_installation($request) {
        try {
            if (!function_exists('hellopack_client')) {
                error_log('HelloPack Simple Installer: HelloPack client function not found');
                return new WP_Error('hellopack_missing', 'HelloPack client is not installed or activated');
            }

            // Include required files for filesystem operations
            require_once(ABSPATH . 'wp-admin/includes/file.php');
            require_once(ABSPATH . 'wp-admin/includes/misc.php');
            require_once(ABSPATH . 'wp-admin/includes/class-wp-upgrader.php');
            require_once(ABSPATH . 'wp-admin/includes/class-plugin-upgrader.php');

            $plugin_id = $request['plugin_id'];
            error_log("HelloPack Simple Installer: Starting installation for plugin ID: " . $plugin_id);

            // Ensure we have admin privileges for this operation
            if (!current_user_can('install_plugins')) {
                error_log('HelloPack Simple Installer: Switching to admin user as current user lacks privileges');
                wp_set_current_user(1); // Temporarily switch to admin user
            }

            // Initialize WP_Filesystem
            global $wp_filesystem;
            if (!function_exists('WP_Filesystem')) {
                require_once(ABSPATH . 'wp-admin/includes/file.php');
            }
            WP_Filesystem();

            $client = hellopack_client();
            if (!$client) {
                error_log('HelloPack Simple Installer: Failed to get HelloPack client instance');
                return new WP_Error('client_init_failed', 'Failed to initialize HelloPack client');
            }

            $api = $client->api();
            if (!$api) {
                error_log('HelloPack Simple Installer: Failed to get API instance');
                return new WP_Error('api_init_failed', 'Failed to initialize HelloPack API');
            }

            // Get plugin information from HelloPack
            $items = $client->items();
            if (!$items) {
                error_log('HelloPack Simple Installer: Failed to get items instance');
                return new WP_Error('items_init_failed', 'Failed to initialize HelloPack items');
            }

            $items->set_plugins(true);
            $install = $items->plugins('install');
            if (!is_array($install)) {
                error_log('HelloPack Simple Installer: Failed to get plugins list');
                return new WP_Error('plugins_list_failed', 'Failed to get plugins list from HelloPack');
            }

            error_log('HelloPack Simple Installer: Retrieved plugins list. Count: ' . count($install));

            $api_obj = new stdClass();
            $found = false;

            // Find the plugin in the install list
            foreach ($install as $value) {
                if (absint($value['id']) === absint($plugin_id)) {
                    $api_obj->name = $value['name'];
                    $api_obj->version = $value['version'];
                    $found = true;
                    error_log("HelloPack Simple Installer: Found plugin - Name: {$value['name']}, Version: {$value['version']}");
                    break;
                }
            }

            if (!$found) {
                error_log("HelloPack Simple Installer: Plugin ID $plugin_id not found in available plugins");
                return new WP_Error('plugin_not_found', "Plugin ID $plugin_id not found in HelloPack");
            }

            // Get the download URL from HelloPack
            $download_link = $api->download($plugin_id);
            if (empty($download_link)) {
                error_log('HelloPack Simple Installer: Failed to get download URL');
                return new WP_Error('download_link_missing', 'Could not get download URL from HelloPack');
            }

            error_log('HelloPack Simple Installer: Successfully got download URL');
            $api_obj->download_link = $download_link;

            // Set up the upgrader
            error_log('HelloPack Simple Installer: Starting plugin installation');
            $skin = new Automatic_Upgrader_Skin();  // Changed from WP_Ajax_Upgrader_Skin
            $upgrader = new Plugin_Upgrader($skin);

            // Define the filesystem credentials
            $creds = request_filesystem_credentials(site_url(), '', false, false, null);
            if (!WP_Filesystem($creds)) {
                error_log('HelloPack Simple Installer: Failed to initialize WP_Filesystem');
                return new WP_Error('fs_init_failed', 'Failed to initialize filesystem');
            }

            $result = $upgrader->install($api_obj->download_link);

            if (is_wp_error($result)) {
                error_log('HelloPack Simple Installer: Installation failed - ' . $result->get_error_message());
                return $result;
            }

            // Get the installed plugin file
            $plugin_file = $upgrader->plugin_info();
            if (!empty($plugin_file)) {
                error_log("HelloPack Simple Installer: Activating plugin: " . $plugin_file);
                
                // Include plugin activation function if not available
                if (!function_exists('activate_plugin')) {
                    require_once ABSPATH . 'wp-admin/includes/plugin.php';
                }
                
                // Activate the plugin
                $activation_result = activate_plugin($plugin_file);
                
                if (is_wp_error($activation_result)) {
                    error_log('HelloPack Simple Installer: Activation failed - ' . $activation_result->get_error_message());
                    return new WP_Error(
                        'activation_failed',
                        sprintf('Plugin installed but activation failed: %s', $activation_result->get_error_message())
                    );
                }
                
                error_log("HelloPack Simple Installer: Successfully installed and activated {$api_obj->name} version {$api_obj->version}");
                return array(
                    'success' => true,
                    'message' => sprintf('Successfully installed and activated %s version %s', $api_obj->name, $api_obj->version)
                );
            }

            // If we couldn't get the plugin file, return success for installation only
            error_log("HelloPack Simple Installer: Successfully installed {$api_obj->name} version {$api_obj->version} (not activated)");
            return array(
                'success' => true,
                'message' => sprintf('Successfully installed %s version %s', $api_obj->name, $api_obj->version)
            );

        } catch (Exception $e) {
            error_log('HelloPack Simple Installer: Exception caught - ' . $e->getMessage());
            return new WP_Error('installation_failed', $e->getMessage());
        }
    }

    /**
     * Handle the plugin search
     * 
     * Response structure:
     * {
     *   "success": boolean,
     *   "query": string,
     *   "results_count": integer,
     *   "results": [
     *     {
     *       "id": string,          // e.g., "1137004424"
     *       "name": string,        // e.g., "Elementor Pro"
     *       "version": string,     // e.g., "3.27.5"
     *       "description": string  // e.g., "Elevate your designs and unlock the full power of Elementor..."
     *     },
     *     ...
     *   ]
     * }
     */
    public function handle_search($request) {
        try {
            if (!function_exists('hellopack_client')) {
                return new WP_Error('hellopack_missing', 'HelloPack client is not installed or activated');
            }

            $search_query = sanitize_text_field($request['query']);
            
            $client = hellopack_client();
            if (!$client) {
                return new WP_Error('client_init_failed', 'Failed to initialize HelloPack client');
            }

            $items = $client->items();
            if (!$items) {
                return new WP_Error('items_init_failed', 'Failed to initialize HelloPack items');
            }

            $items->set_plugins(true);
            $available_plugins = $items->plugins('install');
            
            if (!is_array($available_plugins)) {
                return new WP_Error('plugins_list_failed', 'Failed to get plugins list from HelloPack');
            }

            // Search through plugins
            $results = array_filter($available_plugins, function($plugin) use ($search_query) {
                $searchable_text = strtolower($plugin['name']);
                if (isset($plugin['description'])) {
                    $searchable_text .= ' ' . strtolower($plugin['description']);
                }
                return strpos($searchable_text, strtolower($search_query)) !== false;
            });

            // Format results - simplified version without install_url
            $formatted_results = array_map(function($plugin) {
                return [
                    'id' => $plugin['id'],
                    'name' => $plugin['name'],
                    'version' => $plugin['version'],
                    'description' => isset($plugin['description']) ? $plugin['description'] : ''
                ];
            }, array_values($results));

            return new WP_REST_Response(array(
                'success' => true,
                'query' => $search_query,
                'results_count' => count($formatted_results),
                'results' => $formatted_results
            ));

        } catch (Exception $e) {
            return new WP_Error('search_failed', $e->getMessage());
        }
    }
}

// Initialize the plugin
new HelloPack_Manager();
