<?php
/**
 * Plugin Name: Forklift
 * Plugin URI: https://kisscsabi.com/forklift
 * Description: Forklift is used to get plugins out of the Warehouse. It adds a new tab to the plugin install screen where you can see all the custom plugins currently available in the Warehouse.
 * Version: 1.2.18
 * Author: Jabybaby
 * Author URI: https://kisscsabi.com
 * Text Domain: forklift
 * Update URI: https://warehouse.wp.dev3.smartbits.hu/plugins/forklift
 */

// If this file is called directly, abort.
if (!defined('ABSPATH')) {
    exit;
}

// Define the main plugin file path constant
if (!defined('Forklift_FILE')) {
    define('Forklift_FILE', __FILE__);
}

// Include the auto-update functionality
require_once plugin_dir_path(__FILE__) . 'auto-updates.php';

// Initialize auto-update functionality
Forklift_Auto_Updates::init();

// Define plugin constants
define('Forklift_VERSION', '1.2.18');
define('Forklift_PLUGIN_DIR', plugin_dir_path(__FILE__));
define('Forklift_PLUGIN_URL', plugin_dir_url(__FILE__));

/**
 * The core plugin class
 */
class Forklift {

    /**
     * The unique instance of the plugin.
     *
     * @var Forklift
     */
    private static $instance;
    
    /**
     * API endpoint URL for the warehouse
     */
    private $api_url;

    /**
     * Gets an instance of the plugin.
     *
     * @return Forklift
     */
    public static function get_instance() {
        if (null === self::$instance) {
            self::$instance = new self();
        }
        return self::$instance;
    }

    /**
     * Constructor.
     */
    private function __construct() {
        // Set API URL from settings
        $this->api_url = get_option('forklift_warehouse_url', 'https://warehouse.wp.dev3.smartbits.hu/api/plugins');
        
        $this->load_dependencies();
        $this->define_admin_hooks();
        $this->define_warehouse_hooks();
    }

    /**
     * Load the required dependencies for this plugin.
     */
    private function load_dependencies() {
        // Include the admin class
        require_once Forklift_PLUGIN_DIR . 'admin/class-custom-admin.php';
    }

    /**
     * Register all of the hooks related to the admin area functionality.
     */
    private function define_admin_hooks() {
        $plugin_admin = new Forklift_Admin('forklift', '1.2.18');
        
        // Add admin menu
        add_action('admin_menu', array($plugin_admin, 'add_plugin_admin_menu'));
                
        // Register admin styles and scripts
        add_action('admin_enqueue_scripts', array($plugin_admin, 'enqueue_styles'));
    }
    
    /**
     * Register hooks for the SmartBits Warehouse functionality
     */
    private function define_warehouse_hooks() {
        // Add filter to modify plugin install tabs
        add_filter('install_plugins_tabs', array($this, 'add_custom_tab'));
        
        // Add custom content for our tab
        add_action('install_plugins_smartbits', array($this, 'display_smartbits_tab_content'));
        
        // Load admin assets
        add_action('admin_enqueue_scripts', array($this, 'load_admin_scripts'));
        
        // Add AJAX handler for plugin installation
        add_action('wp_ajax_smartbits_install_plugin', array($this, 'handle_plugin_installation'));
        
        // Add AJAX handler for plugin activation
        add_action('wp_ajax_smartbits_activate_plugin', array($this, 'handle_plugin_activation'));
    }
    
    /**
     * Add custom tab to plugin install screen
     */
    public function add_custom_tab($tabs) {
        $tabs['smartbits'] = __('SmartBits Warehouse', 'forklift');
        return $tabs;
    }
    
    /**
     * Display content for our custom tab
     */
    public function display_smartbits_tab_content() {
        // Get plugins from API
        $plugins = $this->get_plugins_from_api();
        ?>
        <div class="smartbits-warehouse-container">
            <h2><?php _e('SmartBits Plugin Warehouse', 'forklift'); ?></h2>
            <p><?php _e('Browse and install plugins from the SmartBits repository.', 'forklift'); ?></p>
            
            <div class="wp-list-table widefat plugin-install">
                <div id="the-list">
                    <?php 
                    if (!empty($plugins)) {
                        foreach ($plugins as $plugin) {
                            $this->display_plugin_card($plugin);
                        }
                    } else {
                        echo '<div class="notice notice-error"><p>' . __('Unable to fetch plugins from the SmartBits repository.', 'forklift') . '</p></div>';
                    }
                    ?>
                </div>
            </div>
        </div>
        <?php
    }
    
    /**
     * Get plugins from the SmartBits API
     */
    private function get_plugins_from_api() {
        $response = wp_remote_get($this->api_url);
        
        if (is_wp_error($response)) {
            return array();
        }
        
        $body = wp_remote_retrieve_body($response);
        $plugins = json_decode($body, true);
        
        if (empty($plugins) || !is_array($plugins)) {
            return array();
        }
        
        return $plugins;
    }
    
    /**
     * Check if a plugin is installed and get its status
     * 
     * @param string $slug The plugin slug to check
     * @return array Status information (installed, active, file)
     */
    private function get_plugin_status($slug) {
        // Get all installed plugins
        if (!function_exists('get_plugins')) {
            require_once ABSPATH . 'wp-admin/includes/plugin.php';
        }
        
        $all_plugins = get_plugins();
        $status = array(
            'installed' => false,
            'active' => false,
            'file' => '',
        );
        
        // Check if the plugin is installed
        foreach ($all_plugins as $plugin_file => $plugin_data) {
            // Check by slug match (directory name)
            $plugin_dir = dirname($plugin_file);
            if ($plugin_dir === $slug || $plugin_file === $slug) {
                $status['installed'] = true;
                $status['file'] = $plugin_file;
                $status['active'] = is_plugin_active($plugin_file);
                break;
            }
        }
        
        return $status;
    }
    
    /**
     * Display a plugin card
     */
    private function display_plugin_card($plugin) {
        // Get the first plugin info from each plugin entry
        $plugin_key = key($plugin);
        if ($plugin_key === 'name') {
            $plugin_key = key(array_slice($plugin, 1, 1, true));
        }
        
        $plugin_info = isset($plugin[$plugin_key]) ? $plugin[$plugin_key] : array();
        
        if (empty($plugin_info)) {
            return;
        }
        
        // Extract plugin details
        $slug = isset($plugin_info['slug']) ? $plugin_info['slug'] : '';
        
        // Get plugin name - prioritize inner object name, fall back to outer name, then to slug
        $plugin_name = isset($plugin_info['name']) ? $plugin_info['name'] : '';
        if (empty($plugin_name)) {
            $plugin_name = isset($plugin['name']) ? $plugin['name'] : $slug;
        }
        
        // Get plugin description - use description or default to "No description"
        $plugin_description = isset($plugin_info['description']) ? $plugin_info['description'] : __('No description', 'forklift');
        
        $version = isset($plugin_info['new_version']) ? $plugin_info['new_version'] : '';
        $url = isset($plugin_info['url']) ? $plugin_info['url'] : '';
        $package = isset($plugin_info['package']) ? $plugin_info['package'] : '';
        $icon = isset($plugin_info['icons']['1x']) ? $plugin_info['icons']['1x'] : Forklift_PLUGIN_URL . 'admin/images/icon.png';
        $requires = isset($plugin_info['requires']) ? $plugin_info['requires'] : '';
        $tested = isset($plugin_info['tested']) ? $plugin_info['tested'] : '';
        $requires_php = isset($plugin_info['requires_php']) ? $plugin_info['requires_php'] : '';
        
        // Check compatibility
        $wp_version = get_bloginfo('version');
        $compatible = version_compare($wp_version, $requires, '>=');
        $compatibility_text = $compatible ? __('Compatible with your version of WordPress', 'forklift') : sprintf(__('Requires WordPress %s', 'forklift'), $requires);
        $compatibility_class = $compatible ? 'compatible' : 'incompatible';
        
        // Check if plugin is already installed
        $plugin_status = $this->get_plugin_status($slug);
        ?>
        <div class="plugin-card plugin-card-<?php echo esc_attr($slug); ?>">
            <div class="plugin-card-top">
                <div class="name column-name">
                    <h3>
                        <a href="<?php echo esc_url($url); ?>" class="thickbox open-plugin-details-modal" target="_blank">
                            <?php echo esc_html($plugin_name); ?>
                            <img src="<?php echo esc_url($icon); ?>" class="plugin-icon" alt="">
                        </a>
                    </h3>
                </div>
                <div class="action-links">
                    <ul class="plugin-action-buttons">
                        <li>
                            <?php if ($plugin_status['installed']): ?>
                                <?php if ($plugin_status['active']): ?>
                                    <button class="button button-disabled" disabled="disabled">
                                        <?php _e('Active', 'forklift'); ?>
                                    </button>
                                <?php else: ?>
                                    <a class="button activate-now"
                                       href="#" 
                                       data-slug="<?php echo esc_attr($slug); ?>" 
                                       data-name="<?php echo esc_attr($plugin_name); ?>"
                                       data-plugin-file="<?php echo esc_attr($plugin_status['file']); ?>"
                                       aria-label="<?php echo esc_attr(sprintf(__('Activate %s', 'forklift'), $plugin_name)); ?>">
                                        <?php _e('Activate', 'forklift'); ?>
                                    </a>
                                <?php endif; ?>
                            <?php else: ?>
                                <a class="install-now button"
                                   href="#" 
                                   data-slug="<?php echo esc_attr($slug); ?>" 
                                   data-name="<?php echo esc_attr($plugin_name); ?>"
                                   data-plugin-url="<?php echo esc_url($package); ?>"
                                   aria-label="<?php echo esc_attr(sprintf(__('Install %s now', 'forklift'), $plugin_name)); ?>">
                                    <?php _e('Install Now', 'forklift'); ?>
                                </a>
                            <?php endif; ?>
                        </li>
                        <li>
                            <a href="<?php echo esc_url($url); ?>" target="_blank" aria-label="<?php echo esc_attr(sprintf(__('More information about %s', 'forklift'), $plugin_name)); ?>">
                                <?php _e('More Details', 'forklift'); ?>
                            </a>
                        </li>
                    </ul>
                </div>
                <div class="desc column-description">
                    <p><?php echo esc_html($plugin_description); ?></p>
                    <p><?php _e('Version:', 'forklift'); ?> <?php echo esc_html($version); ?></p>
                </div>
                <div class="plugin-card-bottom">
                    <div class="vers column-rating">
                        <?php if ($requires): ?>
                            <?php _e('Requires WordPress:', 'forklift'); ?> <?php echo esc_html($requires); ?>
                        <?php endif; ?>
                    </div>
                    <div class="column-updated">
                        <?php if ($tested): ?>
                            <?php _e('Tested up to:', 'forklift'); ?> <?php echo esc_html($tested); ?>
                        <?php endif; ?>
                    </div>
                    <div class="column-downloaded">
                        <?php if ($requires_php): ?>
                            <?php _e('Requires PHP:', 'forklift'); ?> <?php echo esc_html($requires_php); ?>
                        <?php endif; ?>
                    </div>
                    <div class="column-compatibility">
                        <span class="compatibility-<?php echo $compatibility_class; ?>">
                            <?php echo $compatibility_text; ?>
                        </span>
                    </div>
                </div>
            </div>
        </div>
        <?php
    }
    
    /**
     * Handle plugin installation via AJAX
     */
    public function handle_plugin_installation() {
        try {
            // Verify nonce
            if (!check_ajax_referer('smartbits-warehouse-nonce', 'nonce', false)) {
                wp_send_json_error(array('message' => __('Security check failed.', 'forklift')));
                return;
            }
            
            // Check user capabilities
            if (!current_user_can('install_plugins')) {
                wp_send_json_error(array('message' => __('You do not have permission to install plugins.', 'forklift')));
                return;
            }
            
            // Get plugin details
            $plugin_id = isset($_POST['plugin_id']) ? sanitize_text_field($_POST['plugin_id']) : '';
            $plugin_url = isset($_POST['plugin_url']) ? esc_url_raw($_POST['plugin_url']) : '';
            
            if (empty($plugin_id) || empty($plugin_url)) {
                wp_send_json_error(array('message' => __('Invalid plugin information.', 'forklift')));
                return;
            }
            
            // Check if plugin is already installed
            $plugin_status = $this->get_plugin_status($plugin_id);
            if ($plugin_status['installed']) {
                if ($plugin_status['active']) {
                    wp_send_json_success(array(
                        'message' => __('Plugin is already installed and active.', 'forklift'),
                        'pluginFile' => $plugin_status['file']
                    ));
                    return;
                } else {
                    // Activate the plugin
                    $activate = activate_plugin($plugin_status['file']);
                    
                    if (is_wp_error($activate)) {
                        wp_send_json_error(array(
                            'message' => __('Plugin is installed but failed to activate.', 'forklift') . ' ' . $activate->get_error_message(),
                            'pluginFile' => $plugin_status['file']
                        ));
                        return;
                    }
                    
                    wp_send_json_success(array(
                        'message' => __('Plugin was already installed and has been activated.', 'forklift'),
                        'pluginFile' => $plugin_status['file']
                    ));
                    return;
                }
            }
            
            // Include necessary files for installation
            require_once ABSPATH . 'wp-admin/includes/class-wp-upgrader.php';
            require_once ABSPATH . 'wp-admin/includes/plugin-install.php';
            require_once ABSPATH . 'wp-admin/includes/file.php';
            require_once ABSPATH . 'wp-admin/includes/misc.php';
            
            // Ensure the destination directory is writable
            $upload_dir = wp_upload_dir();
            if (!wp_is_writable($upload_dir['basedir'])) {
                wp_send_json_error(array('message' => __('WordPress uploads directory is not writable.', 'forklift')));
                return;
            }
            
            // Create a new plugin upgrader with our custom skin
            $skin = new WP_Ajax_Upgrader_Skin();
            $upgrader = new Plugin_Upgrader($skin);
            
            // Install the plugin
            $result = $upgrader->install($plugin_url);
            
            // Check for errors
            if (is_wp_error($result)) {
                wp_send_json_error(array('message' => $result->get_error_message()));
                return;
            } elseif (is_wp_error($skin->result)) {
                wp_send_json_error(array('message' => $skin->result->get_error_message()));
                return;
            } elseif ($skin->get_errors()->has_errors()) {
                $error_messages = $skin->get_error_messages();
                if (is_array($error_messages)) {
                    $error_messages = implode(', ', $error_messages);
                }
                wp_send_json_error(array('message' => $error_messages));
                return;
            } elseif (is_null($result)) {
                wp_send_json_error(array('message' => __('Plugin installation failed for an unknown reason.', 'forklift')));
                return;
            }
            
            // Get the plugin basename
            $plugin_basename = $upgrader->plugin_info();
            if (empty($plugin_basename)) {
                wp_send_json_error(array('message' => __('Unable to determine installed plugin file.', 'forklift')));
                return;
            }
            
            // Activate the plugin
            $activate = activate_plugin($plugin_basename);
            
            if (is_wp_error($activate)) {
                wp_send_json_error(array(
                    'message' => __('Plugin installed but failed to activate.', 'forklift') . ' ' . $activate->get_error_message(),
                    'pluginFile' => $plugin_basename
                ));
                return;
            }
            
            wp_send_json_success(array(
                'message' => __('Plugin installed and activated successfully.', 'forklift'),
                'pluginName' => $plugin_id,
                'pluginFile' => $plugin_basename
            ));
            
        } catch (Exception $e) {
            wp_send_json_error(array('message' => __('Error: ', 'forklift') . $e->getMessage()));
        }
    }
    
    /**
     * Handle plugin activation via AJAX
     */
    public function handle_plugin_activation() {
        try {
            // Verify nonce
            if (!check_ajax_referer('smartbits-warehouse-nonce', 'nonce', false)) {
                wp_send_json_error(array('message' => __('Security check failed.', 'forklift')));
                return;
            }
            
            // Check user capabilities
            if (!current_user_can('activate_plugins')) {
                wp_send_json_error(array('message' => __('You do not have permission to activate plugins.', 'forklift')));
                return;
            }
            
            // Get plugin file
            $plugin_file = isset($_POST['plugin_file']) ? sanitize_text_field($_POST['plugin_file']) : '';
            
            if (empty($plugin_file)) {
                wp_send_json_error(array('message' => __('Invalid plugin file.', 'forklift')));
                return;
            }
            
            // Check if the plugin file exists
            if (!file_exists(WP_PLUGIN_DIR . '/' . $plugin_file)) {
                wp_send_json_error(array('message' => __('Plugin file does not exist.', 'forklift')));
                return;
            }
            
            // Check if the plugin is already active
            if (is_plugin_active($plugin_file)) {
                wp_send_json_success(array('message' => __('Plugin is already active.', 'forklift')));
                return;
            }
            
            // Include necessary files
            if (!function_exists('activate_plugin')) {
                require_once ABSPATH . 'wp-admin/includes/plugin.php';
            }
            
            // Activate the plugin
            $result = activate_plugin($plugin_file);
            
            if (is_wp_error($result)) {
                wp_send_json_error(array(
                    'message' => __('Failed to activate plugin.', 'forklift') . ' ' . $result->get_error_message()
                ));
                return;
            }
            
            wp_send_json_success(array(
                'message' => __('Plugin activated successfully.', 'forklift')
            ));
            
        } catch (Exception $e) {
            wp_send_json_error(array('message' => __('Error: ', 'forklift') . $e->getMessage()));
        }
    }
    
    /**
     * Load admin scripts and styles
     */
    public function load_admin_scripts($hook) {
        if ($hook !== 'plugin-install.php') {
            return;
        }

        wp_enqueue_script(
            'smartbits-warehouse-admin-js',
            Forklift_PLUGIN_URL . 'admin/js/admin.js',
            array('jquery', 'updates'),
            Forklift_VERSION,
            true
        );

        wp_localize_script('smartbits-warehouse-admin-js', 'smartbitsWarehouse', array(
            'ajaxUrl' => admin_url('admin-ajax.php'),
            'nonce' => wp_create_nonce('smartbits-warehouse-nonce'),
        ));
    }
}

// Initialize the plugin
Forklift::get_instance(); 