(function($) {
    'use strict';

    $(document).ready(function() {
        // Define fallback strings in case wp.updates.l10n is not available
        var l10n = {
            installing: 'Installing...',
            installed: 'Installed!',
            installNow: 'Install Now',
            installFailedShort: 'Failed',
            activating: 'Activating...',
            active: 'Active',
            activate: 'Activate',
            installFailed: 'Installation failed'
        };

        // Use WordPress strings if available, otherwise use our fallbacks
        if (typeof wp !== 'undefined' && wp.updates && wp.updates.l10n) {
            l10n.installing = wp.updates.l10n.installing || l10n.installing;
            l10n.installed = wp.updates.l10n.installed || l10n.installed;
            l10n.installNow = wp.updates.l10n.installNow || l10n.installNow;
            l10n.installFailedShort = wp.updates.l10n.installFailedShort || l10n.installFailedShort;
            l10n.installFailed = wp.updates.l10n.installFailed || l10n.installFailed;
        }

        // Handle plugin installation
        $('.smartbits-warehouse-container .install-now').on('click', function(e) {
            e.preventDefault();
            
            var button = $(this);
            var slug = button.data('slug');
            var name = button.data('name');
            var pluginUrl = button.data('plugin-url');
            
            // Use reliable text strings with fallbacks
            button.addClass('updating-message').text(l10n.installing);
            // Use a safer approach for aria-label
            button.attr('aria-label', l10n.installing + ' ' + name);
            
            // Custom AJAX for SmartBits repo plugins
            $.ajax({
                url: smartbitsWarehouse.ajaxUrl,
                type: 'POST',
                dataType: 'json',
                data: {
                    action: 'smartbits_install_plugin',
                    plugin_id: slug,
                    plugin_url: pluginUrl,
                    nonce: smartbitsWarehouse.nonce
                },
                success: function(response) {
                    if (response.success) {
                        button.removeClass('updating-message').addClass('updated-message')
                              .text(l10n.installed);
                        button.attr('aria-label', l10n.installed + ' ' + name);
                        
                        // Show success message
                        showNotice('success', response.data.message);
                        
                        // Change button to Active (already activated during install)
                        setTimeout(function() {
                            button.removeClass('updated-message install-now button-primary')
                                  .addClass('button-disabled')
                                  .text('Active')
                                  .attr('disabled', 'disabled');
                        }, 1000);
                    } else {
                        button.removeClass('updating-message')
                              .text(l10n.installFailedShort);
                        
                        // Show error message
                        showNotice('error', response.data.message || l10n.installFailed);
                        
                        // Reset button after some time
                        setTimeout(function() {
                            button.text(l10n.installNow);
                        }, 1000);
                    }
                },
                error: function(xhr, status, error) {
                    button.removeClass('updating-message')
                          .text(l10n.installFailedShort);
                    
                    // Show error message with more details
                    var errorMessage = 'Error: ';
                    if (xhr.responseJSON && xhr.responseJSON.message) {
                        errorMessage += xhr.responseJSON.message;
                    } else {
                        errorMessage += error || 'An unknown error occurred';
                    }
                    
                    showNotice('error', errorMessage);
                    
                    // Reset button after some time
                    setTimeout(function() {
                        button.text(l10n.installNow);
                    }, 1000);
                }
            });
        });
        
        // Handle plugin activation
        $('.smartbits-warehouse-container .activate-now').on('click', function(e) {
            e.preventDefault();
            
            var button = $(this);
            var name = button.data('name');
            var pluginFile = button.data('plugin-file');
            
            // Update button UI
            button.addClass('updating-message').text(l10n.activating);
            
            // Custom AJAX for plugin activation
            $.ajax({
                url: smartbitsWarehouse.ajaxUrl,
                type: 'POST',
                dataType: 'json',
                data: {
                    action: 'smartbits_activate_plugin',
                    plugin_file: pluginFile,
                    nonce: smartbitsWarehouse.nonce
                },
                success: function(response) {
                    if (response.success) {
                        // Show success message
                        showNotice('success', response.data.message);
                        
                        // Change button to Active
                        button.removeClass('updating-message activate-now')
                              .addClass('button-disabled')
                              .text(l10n.active)
                              .attr('disabled', 'disabled');
                    } else {
                        // Show error message
                        showNotice('error', response.data.message || 'Failed to activate plugin.');
                        
                        // Reset button
                        button.removeClass('updating-message').text(l10n.activate);
                    }
                },
                error: function(xhr, status, error) {
                    // Show detailed error message
                    var errorMessage = 'Error activating plugin: ';
                    if (xhr.responseJSON && xhr.responseJSON.message) {
                        errorMessage += xhr.responseJSON.message;
                    } else {
                        errorMessage += error || 'An unknown error occurred';
                    }
                    
                    showNotice('error', errorMessage);
                    
                    // Reset button
                    button.removeClass('updating-message').text(l10n.activate);
                }
            });
        });
        
        // Function to show notices
        function showNotice(type, message) {
            var noticeClass = (type === 'error') ? 'notice-error' : 'notice-success';
            var notice = $('<div class="notice ' + noticeClass + ' is-dismissible"><p>' + message + '</p></div>');
            
            // Remove any existing notices
            $('.smartbits-warehouse-container .notice').remove();
            
            // Add the notice at the top of the container
            $('.smartbits-warehouse-container').prepend(notice);
            
            // Make it dismissible using WordPress core JS
            if (typeof wp !== 'undefined' && wp.updates && wp.updates.addDismissButton) {
                wp.updates.addDismissButton(notice);
            } else {
                // Fallback
                notice.append('<button type="button" class="notice-dismiss"><span class="screen-reader-text">Dismiss this notice.</span></button>');
                notice.find('.notice-dismiss').on('click', function() {
                    notice.fadeOut(300, function() {
                        notice.remove();
                    });
                });
            }
        }
    });
})(jQuery); 