<?php

/**
 * Provide a admin area view for the plugin's settings page
 *
 * This file is used to markup the admin-facing aspects of the plugin.
 *
 * @since      1.0.0
 */

// Set up the header parameters
$page_title = 'Hook viewer';
$page_description = 'Welcome to Hook Viewer. This plugin helps you inspect WordPress hooks and filters.';



function get_function_body($function) {
    // Return if function name is invalid
    if (!is_string($function) || empty($function) || !function_exists($function)) {
        return "Function body not available";
    }
    
    try {
        $func = new ReflectionFunction($function);
        $filename = $func->getFileName();
        $start_line = $func->getStartLine() - 1; // it's actually - 1, otherwise you wont get the function() block
        $end_line = $func->getEndLine();
        $length = $end_line - $start_line;

        if ($filename && file_exists($filename)) {
            $source = file($filename);
            $body = implode("", array_slice($source, $start_line, $length));
            return htmlspecialchars($body);
        }
        return "Source file not found";
    } catch (Exception $e) {
        return "Error retrieving function body: " . $e->getMessage();
    }
}

// Add AJAX endpoint for refreshing hooks (keeping for backward compatibility)
function hook_viewer_refresh_hooks() {
    // Check user capabilities
    if (!current_user_can('manage_options')) {
        wp_die('You do not have permission to access this data', 403);
        return;
    }
    
    global $wp_filter;
    
    // Redirect to the new AJAX endpoint
    hook_viewer_get_hooks_data();
}
add_action('wp_ajax_refresh_hooks', 'hook_viewer_refresh_hooks');


// Create the admin page
function hook_viewer_page() {
    global $wp_filter;
    
    // Generate hook data directly for immediate use without AJAX
    $direct_hooks_data = array();
    
    if (isset($wp_filter) && (is_array($wp_filter) || is_object($wp_filter))) {
        foreach ($wp_filter as $hook_name => $hook_details) {
            $callbacks_data = array();
            $callbacks_count = 0;
            
            if (is_object($hook_details) && property_exists($hook_details, 'callbacks')) {
                foreach ($hook_details->callbacks as $priority => $callbacks) {
                    if (!empty($callbacks) && is_array($callbacks)) {
                        foreach ($callbacks as $callback_id => $callback) {
                            $callback_info = array(
                                'priority' => $priority
                            );
                            
                            // Get accepted args
                            if (isset($callback['accepted_args'])) {
                                $callback_info['accepted_args'] = $callback['accepted_args'];
                            }
                            
                            // Process function
                            if (isset($callback['function'])) {
                                if (is_string($callback['function'])) {
                                    // Regular function
                                    $callback_info['type'] = 'function';
                                    $callback_info['function'] = $callback['function'];
                                    $callback_info['source'] = get_function_body($callback['function']);
                                } else if (is_array($callback['function'])) {
                                    // Class method
                                    $callback_info['type'] = 'class';
                                    if (is_object($callback['function'][0])) {
                                        $callback_info['class'] = get_class($callback['function'][0]);
                                    } else {
                                        $callback_info['class'] = is_string($callback['function'][0]) ? $callback['function'][0] : 'Unknown';
                                    }
                                    $callback_info['method'] = isset($callback['function'][1]) ? $callback['function'][1] : 'Unknown';
                                } else {
                                    // Closure or something else
                                    $callback_info['type'] = 'unknown';
                                    $callback_info['function'] = 'Closure or unknown function type';
                                }
                            } else {
                                $callback_info['type'] = 'invalid';
                                $callback_info['function'] = 'Invalid callback (no function defined)';
                            }
                            
                            $callbacks_data[] = $callback_info;
                            $callbacks_count++;
                        }
                    }
                }
            }
            
            $direct_hooks_data[] = array(
                'name' => $hook_name,
                'callbacks_count' => $callbacks_count,
                'callbacks' => $callbacks_data
            );
        }
    }
    // Convert to JSON for direct use in JS
    $direct_hooks_json = json_encode($direct_hooks_data);
    
    ?>
    <div class="wrap">
        <h1>WordPress Hooks Viewer</h1>
        <p>This tool allows you to explore all registered WordPress hooks, filters, and their callbacks.</p>
        
        <!-- Search and filtering controls -->
        <div class="tablenav top" style="margin: 15px 0;">
            <div class="alignleft actions">
                <input type="text" id="hook-search" placeholder="Search hooks..." style="padding: 5px; width: 250px;">
                <select id="hooks-per-page" class="hooks-per-page" style="padding-top: 5px; padding-bottom: 5px;">
                    <option value="10" selected>10 per page</option>
                    <option value="20">20 per page</option>
                    <option value="50">50 per page</option>
                    <option value="100">100 per page</option>
                    <option value="200">200 per page</option>
                </select>
            </div>
            <div class="tablenav-pages">
                <span class="displaying-num" id="hooks-count">0 items</span>
            </div>
            <br class="clear">
        </div>

        <!-- Hook data table -->
        <table id="hooks-table" class="wp-list-table widefat fixed striped">
            <thead>
                <tr>
                    <th scope="col" class="manage-column column-primary">Hook Name</th>
                    <th scope="col" class="manage-column">Callbacks Count</th>
                    <th scope="col" class="manage-column">Details</th>
                </tr>
            </thead>
            <tbody id="hooks-list">
                <!-- Hook data will be loaded here -->
                <tr>
                    <td colspan="3">Loading hooks data...</td>
                </tr>
            </tbody>
            <tfoot>
                <tr>
                    <th scope="col" class="manage-column column-primary">Hook Name</th>
                    <th scope="col" class="manage-column">Callbacks Count</th>
                    <th scope="col" class="manage-column">Details</th>
                </tr>
            </tfoot>
        </table>

        <!-- Pagination -->
        <div class="tablenav bottom">
            <div class="tablenav-pages" id="hooks-pagination">
                <span class="pagination-links">
                    <button class="first-page button" aria-hidden="true">«</button>
                    <button class="prev-page button" aria-hidden="true">‹</button>
                    <span class="paging-input">
                        <span class="current-page">1</span> of <span class="total-pages">1</span>
                    </span>
                    <button class="next-page button" aria-hidden="true">›</button>
                    <button class="last-page button" aria-hidden="true">»</button>
                </span>
            </div>
        </div>

        <!-- Hook details modal -->
        <div id="hook-details-modal" style="display: none; position: fixed; z-index: 100; left: 0; top: 0; width: 100%; height: 100%; overflow: auto; background-color: rgba(0,0,0,0.4);">
            <div style="background-color: #fefefe; margin: 10% auto; padding: 20px; border: 1px solid #888; width: 80%; max-width: 800px; border-radius: 5px; box-shadow: 0 4px 8px 0 rgba(0,0,0,0.2);">
                <span id="close-modal" style="color: #aaa; float: right; font-size: 28px; font-weight: bold; cursor: pointer;">&times;</span>
                <h2 id="modal-hook-name"></h2>
                <div id="modal-hook-details"></div>
            </div>
        </div>
    </div>

    <script type="text/javascript">
    jQuery(document).ready(function($) {
        // Variables
        let allHooks = [];
        const defaultItemsPerPage = 10;
        let itemsPerPage = defaultItemsPerPage;
        let currentPage = 1;
        let filteredHooks = [];
        
        // Direct loaded hooks data
        const directHooksData = <?php echo $direct_hooks_json; ?>;
        
        // Items per page change handler
        $('#hooks-per-page').on('change', function() {
            itemsPerPage = parseInt($(this).val(), 10);
            currentPage = 1; // Reset to first page
            renderHooksList(currentPage);
            setupPagination();
        });
        
        // Load direct hooks data immediately
        loadDirectHooks();
        
        // Search functionality
        $('#hook-search').on('input', function() {
            const searchTerm = $(this).val().toLowerCase();
            filterAndDisplayHooks(searchTerm);
        });
        
        // Load direct hooks data
        function loadDirectHooks() {
            if (directHooksData && directHooksData.length > 0) {
                console.log('Using direct hooks data:', directHooksData);
                allHooks = directHooksData;
                filteredHooks = [...allHooks];
                updateHooksCount(filteredHooks.length);
                renderHooksList(1);
                setupPagination();
            } else {
                $('#hooks-list').html('<tr><td colspan="3">No hooks found in direct data</td></tr>');
            }
        }
        
        // Filter and display hooks
        function filterAndDisplayHooks(searchTerm) {
            if (!searchTerm) {
                filteredHooks = [...allHooks];
            } else {
                filteredHooks = allHooks.filter(hook => 
                    hook.name.toLowerCase().includes(searchTerm)
                );
            }
            
            currentPage = 1;
            updateHooksCount(filteredHooks.length);
            renderHooksList(currentPage);
            setupPagination();
        }
        
        // Render hooks list for a specific page
        function renderHooksList(page) {
            currentPage = page; // Ensure current page is updated
            const startIndex = (page - 1) * itemsPerPage;
            const endIndex = startIndex + itemsPerPage;
            const pageHooks = filteredHooks.slice(startIndex, endIndex);
            
            if (pageHooks.length === 0) {
                $('#hooks-list').html('<tr><td colspan="3">No hooks found</td></tr>');
                return;
            }
            
            let html = '';
            pageHooks.forEach(hook => {
                html += `
                <tr>
                    <td class="column-primary">${hook.name}</td>
                    <td>${hook.callbacks_count}</td>
                    <td><button class="button view-hook-details" data-hook="${hook.name}">View Details</button></td>
                </tr>`;
            });
            
            $('#hooks-list').html(html);
            
            // Attach event listeners to the view details buttons
            $('.view-hook-details').on('click', function() {
                const hookName = $(this).data('hook');
                showHookDetails(hookName);
            });
            
            // Update pagination display after rendering
            $('.current-page').text(currentPage);
        }
        
        // Show hook details in modal
        function showHookDetails(hookName) {
            const hook = allHooks.find(h => h.name === hookName);
            if (!hook) return;
            
            $('#modal-hook-name').text(hookName);
            
            let detailsHtml = '';
            
            if (hook.callbacks && hook.callbacks.length > 0) {
                hook.callbacks.forEach(callback => {
                    detailsHtml += `
                    <div style="margin-bottom: 20px; padding: 15px; background: #f8f8f8; border-radius: 5px;">
                        <h3>Priority: ${callback.priority}</h3>`;
                    
                    if (callback.type === 'function') {
                        detailsHtml += `<p><strong>Function:</strong> ${callback.function}</p>`;
                    } else if (callback.type === 'class') {
                        detailsHtml += `
                        <p><strong>Class:</strong> ${callback.class}</p>
                        <p><strong>Method:</strong> ${callback.method}</p>`;
                    } else {
                        detailsHtml += `<p><strong>Type:</strong> ${callback.type}</p>`;
                    }
                    
                    if (callback.accepted_args) {
                        detailsHtml += `<p><strong>Accepted Arguments:</strong> ${callback.accepted_args}</p>`;
                    }
                    
                    if (callback.source) {
                        detailsHtml += `
                        <div>
                            <strong>Source:</strong>
                            <pre style="background: #f1f1f1; padding: 10px; overflow: auto; max-height: 200px; border-radius: 4px; font-size: 12px;">${callback.source}</pre>
                        </div>`;
                    }
                    
                    detailsHtml += '</div>';
                });
            } else {
                detailsHtml = `
                <div style="margin-bottom: 20px; padding: 15px; background: #f8f8f8; border-radius: 5px;">
                    <p><strong>Hook Name:</strong> ${hook.name}</p>
                    <p><strong>Callbacks Count:</strong> ${hook.callbacks_count}</p>
                    <p>No detailed callback information available.</p>
                </div>`;
            }
            
            $('#modal-hook-details').html(detailsHtml);
            $('#hook-details-modal').show();
        }
        
        // Close modal
        $('#close-modal').on('click', function() {
            $('#hook-details-modal').hide();
        });
        
        // Close modal when clicking outside
        $(window).on('click', function(event) {
            if ($(event.target).is('#hook-details-modal')) {
                $('#hook-details-modal').hide();
            }
        });
        
        // Setup pagination
        function setupPagination() {
            const totalPages = Math.ceil(filteredHooks.length / itemsPerPage);
            $('.current-page').text(currentPage);
            $('.total-pages').text(totalPages);
            
            // Update button states
            updatePaginationButtonStates(totalPages);
            
            // Attach click events to pagination buttons
            setupPaginationClickHandlers(totalPages);
        }
        
        function updatePaginationButtonStates(totalPages) {
            // Reset all buttons first
            $('.first-page, .prev-page, .next-page, .last-page').removeClass('disabled');
            
            // Disable first/prev buttons if on first page
            if (currentPage === 1) {
                $('.first-page, .prev-page').addClass('disabled');
            }
            
            // Disable next/last buttons if on last page or no pages
            if (currentPage === totalPages || totalPages === 0) {
                $('.next-page, .last-page').addClass('disabled');
            }
        }
        
        function setupPaginationClickHandlers(totalPages) {
            // First page button
            $('.first-page').off('click').on('click', function(e) {
                e.preventDefault();
                if (currentPage !== 1 && !$(this).hasClass('disabled')) {
                    currentPage = 1;
                    renderHooksList(currentPage);
                    updatePaginationButtonStates(totalPages);
                }
            });
            
            // Previous page button
            $('.prev-page').off('click').on('click', function(e) {
                e.preventDefault();
                if (currentPage > 1 && !$(this).hasClass('disabled')) {
                    currentPage--;
                    renderHooksList(currentPage);
                    updatePaginationButtonStates(totalPages);
                }
            });
            
            // Next page button
            $('.next-page').off('click').on('click', function(e) {
                e.preventDefault();
                if (currentPage < totalPages && !$(this).hasClass('disabled')) {
                    currentPage++;
                    renderHooksList(currentPage);
                    updatePaginationButtonStates(totalPages);
                }
            });
            
            // Last page button
            $('.last-page').off('click').on('click', function(e) {
                e.preventDefault();
                if (currentPage !== totalPages && totalPages > 0 && !$(this).hasClass('disabled')) {
                    currentPage = totalPages;
                    renderHooksList(currentPage);
                    updatePaginationButtonStates(totalPages);
                }
            });
        }
        
        // Update hooks count display
        function updateHooksCount(count) {
            $('#hooks-count').text(count + ' ' + (count === 1 ? 'item' : 'items'));
        }
        
        // Add CSS for pagination
        $('head').append(`
            <style>
                .pagination-links .button {
                    display: inline-block;
                    text-decoration: none;
                    cursor: pointer;
                    border: 1px solid #ddd;
                    border-radius: 3px;
                    margin: 0 2px;
                    padding: 3px 8px;
                    background: #f7f7f7;
                    color: #0073aa;
                }
                .pagination-links .button.disabled {
                    opacity: 0.5;
                    cursor: default;
                    color: #a0a5aa;
                    background: #f7f7f7;
                }
                .pagination-links .paging-input {
                    margin: 0 5px;
                    vertical-align: middle;
                }
            </style>
        `);
    });
    </script>
    <?php
}

// AJAX handler for getting hooks data
function hook_viewer_get_hooks_data() {
    // Increase memory limit to handle large hook data
    @ini_set('memory_limit', '256M');
    // Set longer timeout
    @set_time_limit(120);
    
    // Start output buffering to catch any unexpected output
    ob_start();
    
    // Log the beginning of the function for debugging
    error_log('hook_viewer_get_hooks_data function started');
    
    // Verify nonce for security
    if (!isset($_POST['_wpnonce'])) {
        error_log('Hook Viewer: Missing nonce in request');
        ob_end_clean();
        wp_send_json_error(array('message' => 'Security check failed - missing nonce'));
        return;
    }
    
    if (!wp_verify_nonce($_POST['_wpnonce'], 'hook_viewer_nonce')) {
        error_log('Hook Viewer: Nonce verification failed');
        ob_end_clean();
        wp_send_json_error(array('message' => 'Security check failed - invalid nonce'));
        return;
    }
    
    // Check user capabilities
    if (!current_user_can('manage_options')) {
        error_log('Hook Viewer: Insufficient permissions');
        ob_end_clean();
        wp_send_json_error(array('message' => 'You do not have permission to access this data'));
        return;
    }
    
    global $wp_filter;
    
    $hooks_data = array();
    
    // Log the status of $wp_filter
    if (empty($wp_filter)) {
        error_log('Hook Viewer: $wp_filter is empty');
    } else if (!is_array($wp_filter) && !is_object($wp_filter)) {
        error_log('Hook Viewer: $wp_filter is not an array or object, it is: ' . gettype($wp_filter));
    } else {
        $count = is_array($wp_filter) ? count($wp_filter) : 'unknown (object)';
        error_log('Hook Viewer: $wp_filter contains ' . $count . ' hooks');
    }
    
    if (empty($wp_filter)) {
        error_log('Hook Viewer: Empty $wp_filter');
        ob_end_clean();
        wp_send_json_error(array('message' => 'No hooks found'));
        return;
    }
    
    if (!is_array($wp_filter) && !is_object($wp_filter)) {
        error_log('Hook Viewer: Invalid $wp_filter type: ' . gettype($wp_filter));
        ob_end_clean();
        wp_send_json_error(array('message' => 'Invalid hook data structure'));
        return;
    }
    
    try {
        // Initialize with an empty array to prevent null responses
        $hooks_data = array();
        
        foreach ($wp_filter as $hook_name => $hook_details) {
            $callbacks_data = array();
            $callbacks_count = 0;
            
            // Skip if hook details is not an object
            if (!is_object($hook_details)) {
                error_log("Hook Viewer: Hook details for '$hook_name' is not an object: " . gettype($hook_details));
                continue;
            }
            
            // Skip if callbacks property doesn't exist
            if (!property_exists($hook_details, 'callbacks')) {
                error_log("Hook Viewer: Invalid hook details for hook: $hook_name (no callbacks property)");
                continue;
            }
            
            if (!empty($hook_details->callbacks)) {
                foreach ($hook_details->callbacks as $priority => $callbacks) {
                    if (!empty($callbacks) && is_array($callbacks)) {
                        foreach ($callbacks as $callback_id => $callback) {
                            try {
                                $callbacks_count++;
                                $callback_data = array(
                                    'priority' => $priority,
                                    'accepted_args' => isset($callback['accepted_args']) ? $callback['accepted_args'] : null
                                );
                                
                                // Check if function element exists
                                if (!isset($callback['function'])) {
                                    $callback_data['type'] = 'invalid';
                                    $callback_data['function'] = 'Missing function reference';
                                    $callbacks_data[] = $callback_data;
                                    continue;
                                }
                                
                                if (is_array($callback['function'])) {
                                    if (is_object($callback['function'][0])) {
                                        $callback_data['type'] = 'class';
                                        try {
                                            $callback_data['class'] = get_class($callback['function'][0]);
                                        } catch (Exception $e) {
                                            $callback_data['class'] = 'Unknown class (error getting class name)';
                                        }
                                        $callback_data['method'] = $callback['function'][1];
                                    } else {
                                        $callback_data['type'] = 'class';
                                        $callback_data['class'] = is_string($callback['function'][0]) ? $callback['function'][0] : 'Unknown class';
                                        $callback_data['method'] = isset($callback['function'][1]) ? $callback['function'][1] : 'Unknown method';
                                    }
                                } else if (is_string($callback['function'])) {
                                    $callback_data['type'] = 'function';
                                    $callback_data['function'] = $callback['function'];
                                    
                                    // Skip source code retrieval to reduce complexity
                                    $callback_data['source'] = 'Source code retrieval disabled for debugging';
                                } else if (is_callable($callback['function'])) {
                                    $callback_data['type'] = 'closure';
                                    $callback_data['function'] = 'Anonymous function or closure';
                                } else {
                                    $callback_data['type'] = 'unknown';
                                    $callback_data['function'] = 'Unknown function type: ' . gettype($callback['function']);
                                }
                                
                                $callbacks_data[] = $callback_data;
                            } catch (Exception $e) {
                                error_log("Hook Viewer: Error processing callback for hook $hook_name: " . $e->getMessage());
                                // Add minimal info for this callback
                                $callbacks_data[] = array(
                                    'type' => 'error',
                                    'priority' => $priority,
                                    'error' => $e->getMessage()
                                );
                            }
                        }
                    }
                }
            }
            
            $hooks_data[] = array(
                'name' => $hook_name,
                'callbacks_count' => $callbacks_count,
                'callbacks' => $callbacks_data
            );
            
            // Check for unexpected output
            $output = ob_get_contents();
            if (!empty($output)) {
                error_log("Hook Viewer: Unexpected output during processing hook $hook_name: $output");
                ob_clean();
            }
        }
        
        error_log('Hook Viewer: Successfully processed ' . count($hooks_data) . ' hooks');
        
        // Check for an unexpected output before sending response
        $unexpected_output = ob_get_clean();
        if (!empty($unexpected_output)) {
            error_log('Hook Viewer: Unexpected output before JSON response: ' . $unexpected_output);
        } else {
            ob_end_clean();
        }
        
        // Send a smaller response for testing - just hook names and callback counts
        $simplified_hooks = array();
        foreach ($hooks_data as $hook) {
            $simplified_hooks[] = array(
                'name' => $hook['name'],
                'callbacks_count' => $hook['callbacks_count']
            );
        }
        
        // Use WordPress's built-in JSON response function
        wp_send_json_success($simplified_hooks);
    } catch (Exception $e) {
        $buffer = ob_get_clean();
        error_log('Hook Viewer: Critical error in processing hooks: ' . $e->getMessage());
        error_log('Hook Viewer: Buffer content at error: ' . $buffer);
        wp_send_json_error(array(
            'message' => 'Error processing hooks data',
            'error' => $e->getMessage(),
            'file' => $e->getFile(),
            'line' => $e->getLine()
        ));
    }
    exit;
}
add_action('wp_ajax_get_hooks_data', 'hook_viewer_get_hooks_data');

// Set the content callback
$content_callback = 'hook_viewer_page';

// Include the header template
include_once plugin_dir_path(__FILE__) . 'custom-header.php';
?> 