<?php
/**
 * Plugin Name: Service Charge Plugin
 * Plugin URI: https://kisscsabi.com/custom-plugin-template
 * Description: This is a template plugin for creating new plugins. You can install it to see a test page and test auto-updates. Whenever you wish to make a new plugin, you can install this plugin and start from there.
 * Version: 1.2.13
 * Author: Jabybaby
 * Author URI: https://kisscsabi.com
 * Text Domain: custom-plugin-template
 * Update URI: https://warehouse.wp.dev3.smartbits.hu/plugins/custom-plugin-template
 */

// If this file is called directly, abort.
if (!defined('ABSPATH')) {
    exit;
}

// Define the main plugin file path constant
if (!defined('Custom_Plugin_Template_FILE')) {
    define('Custom_Plugin_Template_FILE', __FILE__);
}

// Include the auto-update functionality
require_once plugin_dir_path(__FILE__) . 'auto-updates.php';

// Initialize auto-update functionality
Custom_Plugin_Template_Auto_Updates::init();

// Define plugin constants
define('Custom_Plugin_Template_VERSION', '1.2.13');
define('Custom_Plugin_Template_PLUGIN_DIR', plugin_dir_path(__FILE__));
define('Custom_Plugin_Template_PLUGIN_URL', plugin_dir_url(__FILE__));

/**
 * The core plugin class
 */
class Custom_Plugin_Template {

    /**
     * The unique instance of the plugin.
     *
     * @var Custom_Plugin_Template
     */
    private static $instance;

    /**
     * Gets an instance of the plugin.
     *
     * @return Custom_Plugin_Template
     */
    public static function get_instance() {
        if (null === self::$instance) {
            self::$instance = new self();
        }
        return self::$instance;
    }

    /**
     * Constructor.
     */
    private function __construct() {
        $this->load_dependencies();
        $this->define_admin_hooks();
        $this->define_service_charge_hooks();
    }

    /**
     * Load the required dependencies for this plugin.
     */
    private function load_dependencies() {
        // Include the admin class
        require_once Custom_Plugin_Template_PLUGIN_DIR . 'admin/class-custom-admin.php';
    }

    /**
     * Register all of the hooks related to the admin area functionality.
     */
    private function define_admin_hooks() {
        $plugin_admin = new Custom_Plugin_Template_Admin('custom-plugin-template', '1.2.13');
        
        // Add admin menu
        add_action('admin_menu', array($plugin_admin, 'add_plugin_admin_menu'));
                
        // Register admin styles and scripts
        add_action('admin_enqueue_scripts', array($plugin_admin, 'enqueue_styles'));
    }

    /**
     * Register all of the hooks related to the service charge functionality.
     */
    private function define_service_charge_hooks() {
        // Add service charge calculation hook
        add_action('woocommerce_cart_calculate_fees', array($this, 'add_service_charge'));
        
        // Add cart note
        add_filter('woocommerce_cart_totals_order_total_html', array($this, 'add_cart_note'));
        
        // Add service charge field to product data
        add_action('woocommerce_product_options_general_product_data', array($this, 'add_service_charge_field'));
        
        // Save service charge field
        add_action('woocommerce_process_product_meta', array($this, 'save_service_charge_field'));
        
        // Add service charge to quick edit
        add_action('quick_edit_custom_box', array($this, 'add_service_charge_quick_edit'), 10, 2);
        add_action('bulk_edit_custom_box', array($this, 'add_service_charge_quick_edit'), 10, 2);
        
        // Save service charge from quick edit
        add_action('woocommerce_product_quick_edit_save', array($this, 'save_service_charge_quick_edit'));
        add_action('woocommerce_product_bulk_edit_save', array($this, 'save_service_charge_quick_edit'));
        
        // Add service charge column
        add_filter('manage_product_posts_columns', array($this, 'add_service_charge_column'));
        add_action('manage_product_posts_custom_column', array($this, 'display_service_charge_column'), 10, 2);
        
        // Make service charge column sortable
        add_filter('manage_edit-product_sortable_columns', array($this, 'make_service_charge_column_sortable'));
        add_action('pre_get_posts', array($this, 'service_charge_column_orderby'));
    }

    /**
     * Add service charge to cart
     */
    public function add_service_charge(WC_Cart $cart) {
        $language = apply_filters('wpml_current_language', null);
        
        // Create a service charge manager
        $service_charge_manager = new ServiceChargeManager();
        
        $items = $cart->get_cart();
        foreach ($items as $values) {
            $product = $values['data'];

            $price = $product->get_price();
            $quantity = $values['quantity'];
            
            // Check if service charge is enabled for this product
            $has_service_charge = get_post_meta($product->get_id(), '_has_service_charge', true);
            if (!$has_service_charge) continue;
            
            $taxclass = $product->get_tax_class();
            $vatrates = array_values(WC_Tax::get_rates_for_tax_class($taxclass));
            $rate = intval($vatrates[0]->tax_rate);
            
            $gross = $price * $quantity;
            $charge = $gross * 0.15;
            
            // Add the charge to the appropriate VAT rate bucket
            $service_charge_manager->addCharge($rate, $charge / (1 + $rate/100));
        }
        
        // Add all charges as fees
        $service_charge_manager->addChargesAsFees($cart, $language);
        
        return $cart;
    }

    /**
     * Add cart note about service charge
     */
    public function add_cart_note($value) {
        $cart = WC()->cart;
        $language = apply_filters('wpml_current_language', null);
        
        if (count($cart->get_fees()) == 0) {
            return $value;
        }

        $value .= '<br>';

        if ($language == 'en') {
            $value .= '<small>* Including courses, the VAT and the service charge.</small>';
        } else {
            $value .= '<small>* Tartalmazza az ételsort, az áfát és szervizdíjat.</small>';
        }

        return $value;
    }

    /**
     * Add service charge field to product data
     */
    public function add_service_charge_field() {
        global $woocommerce, $post;
        
        echo '<div class="options_group">';
        
        woocommerce_wp_checkbox(array(
            'id' => '_has_service_charge',
            'label' => __('Service Charge', 'woocommerce'),
            'description' => __('Check this box if this product should have a service charge applied.', 'woocommerce')
        ));
        
        echo '</div>';
    }

    /**
     * Save service charge field
     */
    public function save_service_charge_field($post_id) {
        $has_service_charge = isset($_POST['_has_service_charge']) ? 'yes' : 'no';
        update_post_meta($post_id, '_has_service_charge', $has_service_charge);
    }

    /**
     * Add service charge to quick edit
     */
    public function add_service_charge_quick_edit($column_name, $post_type) {
        if ($post_type !== 'product' || $column_name !== 'price') {
            return;
        }
        
        $product = wc_get_product(get_the_ID());
        $has_service_charge = get_post_meta($product->get_id(), '_has_service_charge', true);
        
        echo '<fieldset class="inline-edit-col-right">';
        echo '<div class="inline-edit-col">';
        echo '<label class="inline-edit-group">';
        echo '<input type="checkbox" name="_has_service_charge" value="yes" ' . checked($has_service_charge, 'yes', false) . ' />';
        echo '<span class="checkbox-title">' . __('Service Charge', 'woocommerce') . '</span>';
        echo '</label>';
        echo '</div>';
        echo '</fieldset>';
    }

    /**
     * Save service charge from quick edit
     */
    public function save_service_charge_quick_edit($product) {
        $has_service_charge = isset($_POST['_has_service_charge']) ? 'yes' : 'no';
        update_post_meta($product->get_id(), '_has_service_charge', $has_service_charge);
    }

    /**
     * Add service charge column
     */
    public function add_service_charge_column($columns) {
        $new_columns = array();
        
        foreach ($columns as $key => $value) {
            $new_columns[$key] = $value;
            
            // Add our column after the price column
            if ($key === 'price') {
                $new_columns['service_charge'] = __('Service Charge', 'woocommerce');
            }
        }
        
        return $new_columns;
    }

    /**
     * Display service charge column content
     */
    public function display_service_charge_column($column, $post_id) {
        if ($column === 'service_charge') {
            $has_service_charge = get_post_meta($post_id, '_has_service_charge', true);
            echo $has_service_charge === 'yes' ? '✓' : '—';
        }
    }

    /**
     * Make service charge column sortable
     */
    public function make_service_charge_column_sortable($columns) {
        $columns['service_charge'] = 'service_charge';
        return $columns;
    }

    /**
     * Handle service charge column sorting
     */
    public function service_charge_column_orderby($query) {
        if (!is_admin() || !$query->is_main_query()) {
            return;
        }
        
        $orderby = $query->get('orderby');
        
        if ($orderby === 'service_charge') {
            $query->set('meta_key', '_has_service_charge');
            $query->set('orderby', 'meta_value');
        }
    }
}

/**
 * ServiceChargeManager class to handle service charges for different VAT rates
 */
class ServiceChargeManager {
    /**
     * @var array Stores charges by VAT rate
     */
    private $charges = [];
    
    /**
     * Add a charge for a specific VAT rate
     * 
     * @param int $vat_rate The VAT rate
     * @param float $charge The charge amount
     */
    public function addCharge($vat_rate, $charge) {
        if (!isset($this->charges[$vat_rate])) {
            $this->charges[$vat_rate] = 0;
        }
        $this->charges[$vat_rate] += $charge;
    }
    
    /**
     * Add all charges as fees to the cart
     * 
     * @param WC_Cart $cart The cart object
     * @param string $language The current language
     */
    public function addChargesAsFees($cart, $language) {
        foreach ($this->charges as $vat_rate => $charge) {
            if ($charge > 0) {
                $name = $language == 'en' 
                    ? sprintf('Service charge (%d%%)', $vat_rate) 
                    : sprintf('Szervizdíj: %d%%', $vat_rate);
                
                $cart->add_fee($name, $charge, true, (string)$vat_rate);
            }
        }
    }
}

// Initialize the plugin
Custom_Plugin_Template::get_instance(); 