<?php
/**
 * Auto-update functionality for custom-plugin-template
 */

// Prevent direct access to this file
if (!defined('ABSPATH')) {
    exit;
}


/**
 * Class to handle auto-update functionality
 */

class Custom_Plugin_Template_Auto_Updates {
    
    /**
     * Initialize the auto-update functionality
     */
    public static function init() {
        // Check for updates - higher priority to override other filters
        add_filter('pre_set_site_transient_update_plugins', [self::class, 'force_update_check'], 999, 1);
        
        // Plugin information for the "View details" screen
        add_filter('plugins_api', [self::class, 'plugin_information'], 10, 3);
        
        // Force clear update cache on every admin page load (for testing)
        add_action('admin_init', [self::class, 'clear_update_cache']);
    }
    
    /**
     * Force update check with higher priority
     */
    public static function force_update_check($transient) {
        if (empty($transient) || !is_object($transient)) {
            return $transient;
        }
        
        $plugin_file = plugin_basename(Custom_Plugin_Template_FILE);
        
        // Get current version
        if (!function_exists('get_plugin_data')) {
            require_once(ABSPATH . 'wp-admin/includes/plugin.php');
        }
        $plugin_data = get_plugin_data(Custom_Plugin_Template_FILE);
        $current_version = $plugin_data['Version'];
        
        // Get update data
        $update_data = self::get_update_data();
        
        if ($update_data && isset($update_data['new_version'])) {
            $remote_version = $update_data['new_version'];
            
            // If remote version is higher, force add to the update list
            if (version_compare($remote_version, $current_version, '>')) {
                $item = (object) [
                    'id' => $plugin_file,
                    'slug' => 'custom-plugin-template',
                    'plugin' => $plugin_file,
                    'new_version' => $remote_version,
                    'changelog' => 'Latest version: ' . ($update_data['new_version'] ?? $plugin_data['Version']).'Current version: ' . $current_version.', data dump: ' . json_encode($update_data).', plugin data dump: ' . json_encode($plugin_data).', plugin file: ' . $plugin_file,
                    'url' => $update_data['url'] ?? '',
                    'package' => $update_data['package'] ?? '',
                    'icons' => $update_data['icons'] ?? [],
                    'banners' => $update_data['banners'] ?? [],
                    'requires' => $update_data['requires'] ?? '',
                    'tested' => $update_data['tested'] ?? '',
                    'requires_php' => $update_data['requires_php'] ?? ''
                ];
                
                // Force add to response
                $transient->response[$plugin_file] = $item;
                
                // Remove from no_update if it's there
                if (isset($transient->no_update[$plugin_file])) {
                    unset($transient->no_update[$plugin_file]);
                }
            }
        }
        
        return $transient;
    }
    
    /**
     * Get update data from server
     */
    public static function get_update_data() {
        $plugin_file = Custom_Plugin_Template_FILE;
        $plugin_file = plugin_basename($plugin_file);
        $update_url = 'https://warehouse.wp.dev3.smartbits.hu/plugins/custom-plugin-template';
        
        $response = wp_remote_get($update_url);
        
        if (is_wp_error($response) || wp_remote_retrieve_response_code($response) !== 200) {
            return false;
        }
        
        $body = wp_remote_retrieve_body($response);
        $update_data = json_decode($body, true);
        
        if (json_last_error() !== JSON_ERROR_NONE || empty($update_data[$plugin_file])) {
            return false;
        }
        
        return $update_data[$plugin_file];
    }
    
    /**
     * Provide plugin information for the "View details" screen
     */
    public static function plugin_information($result, $action, $args) {
        if ($action !== 'plugin_information') {
            return $result;
        }
        
        if (!isset($args->slug) || $args->slug !== 'custom-plugin-template') {
            return $result;
        }
        
        if (!function_exists('get_plugin_data')) {
            require_once(ABSPATH . 'wp-admin/includes/plugin.php');
        }
        
        $plugin_data = get_plugin_data($plugin_file);
        $update_data = self::get_update_data();
        
        if (!$update_data) {
            return $result;
        }
        
        // Create a plugin_information object
        $info = new stdClass();
        $info->name = $plugin_data['Name'];
        $info->slug = 'custom-plugin-template';
        $info->version = $update_data['new_version'] ?? $plugin_data['Version'];
        $info->author = $plugin_data['Author'];
        $info->requires = $update_data['requires'] ?? '';
        $info->tested = $update_data['tested'] ?? '';
        $info->requires_php = $update_data['requires_php'] ?? '';
        $info->sections = array(
            'description' => $plugin_data['Description'],
            'changelog' => 'Latest version: ' . ($update_data['new_version'] ?? $plugin_data['Version']).'Current version: ' . $current_version.', data dump: ' . json_encode($update_data).', plugin data dump: ' . json_encode($plugin_data).', plugin file: ' . $plugin_file
        );
        
        return $info;
    }
    
    /**
     * Clear update cache
     */
    public static function clear_update_cache() {
        delete_site_transient('update_plugins');
    }
} 