<?php
/**
 * Generic options page template for the custom plugin system
 *
 * This file provides a reusable template for creating options pages with form fields.
 *
 * @since      1.0.0
 */

// Get the page parameters from the including file
$page_title = isset($page_title) ? $page_title : 'Options';
$page_description = isset($page_description) ? $page_description : 'Configure plugin settings.';
$option_name = isset($option_name) ? $option_name : 'plugin_options';
$option_group = isset($option_group) ? $option_group : 'plugin_options';
$fields = isset($fields) && is_array($fields) ? $fields : array();

// Define the content callback function
function render_options_content() {
    global $option_name, $option_group, $fields;
    
    // Get saved options
    $options = get_option($option_name, array());
    
    // Check if form was submitted
    if (isset($_POST['submit'])) {
        // Verify nonce
        if (isset($_POST['_wpnonce']) && wp_verify_nonce($_POST['_wpnonce'], $option_group . '_options')) {
            // Process form data
            $new_options = array();
            
            foreach ($fields as $field) {
                $field_id = $field['id'];
                $field_type = $field['type'];
                
                if (isset($_POST[$field_id])) {
                    $value = $_POST[$field_id];
                    
                    // Sanitize based on field type
                    switch ($field_type) {
                        case 'text':
                            $value = sanitize_text_field($value);
                            break;
                        case 'number':
                            $value = intval($value);
                            // Apply min/max constraints if defined
                            if (isset($field['min']) && $value < $field['min']) {
                                $value = $field['min'];
                            }
                            if (isset($field['max']) && $value > $field['max']) {
                                $value = $field['max'];
                            }
                            break;
                        case 'textarea':
                            $value = sanitize_textarea_field($value);
                            break;
                        case 'checkbox':
                            $value = (bool) $value;
                            break;
                        default:
                            $value = sanitize_text_field($value);
                    }
                    
                    $new_options[$field_id] = $value;
                } elseif ($field_type === 'checkbox') {
                    // For checkboxes, if not set, it's false
                    $new_options[$field_id] = false;
                }
            }
            
            // Update options
            update_option($option_name, $new_options);
            
            // Show success message
            echo '<div class="notice notice-success is-dismissible"><p>Settings saved.</p></div>';
            
            // Refresh options
            $options = $new_options;
        } else {
            // Show error message
            echo '<div class="notice notice-error is-dismissible"><p>Security check failed. Please try again.</p></div>';
        }
    }
    
    // Output the form
    ?>
    <form method="post" action="">
        <?php wp_nonce_field($option_group . '_options'); ?>
        
        <table class="form-table">
            <tbody>
                <?php foreach ($fields as $field): ?>
                    <tr>
                        <th scope="row">
                            <label for="<?php echo esc_attr($field['id']); ?>"><?php echo esc_html($field['label']); ?></label>
                        </th>
                        <td>
                            <?php
                            $field_id = $field['id'];
                            $field_type = $field['type'];
                            $field_value = isset($options[$field_id]) ? $options[$field_id] : (isset($field['default']) ? $field['default'] : '');
                            
                            switch ($field_type) {
                                case 'text':
                                    ?>
                                    <input type="text" 
                                           id="<?php echo esc_attr($field_id); ?>" 
                                           name="<?php echo esc_attr($field_id); ?>" 
                                           value="<?php echo esc_attr($field_value); ?>" 
                                           class="regular-text"
                                           <?php echo isset($field['placeholder']) ? 'placeholder="' . esc_attr($field['placeholder']) . '"' : ''; ?>
                                    >
                                    <?php
                                    break;
                                    
                                case 'number':
                                    ?>
                                    <input type="number" 
                                           id="<?php echo esc_attr($field_id); ?>" 
                                           name="<?php echo esc_attr($field_id); ?>" 
                                           value="<?php echo esc_attr($field_value); ?>" 
                                           class="regular-text"
                                           <?php echo isset($field['min']) ? 'min="' . esc_attr($field['min']) . '"' : ''; ?>
                                           <?php echo isset($field['max']) ? 'max="' . esc_attr($field['max']) . '"' : ''; ?>
                                           <?php echo isset($field['step']) ? 'step="' . esc_attr($field['step']) . '"' : ''; ?>
                                    >
                                    <?php
                                    break;
                                    
                                case 'textarea':
                                    ?>
                                    <textarea id="<?php echo esc_attr($field_id); ?>" 
                                              name="<?php echo esc_attr($field_id); ?>" 
                                              rows="5" 
                                              class="large-text"
                                              <?php echo isset($field['placeholder']) ? 'placeholder="' . esc_attr($field['placeholder']) . '"' : ''; ?>
                                    ><?php echo esc_textarea($field_value); ?></textarea>
                                    <?php
                                    break;
                                    
                                case 'checkbox':
                                    ?>
                                    <input type="checkbox" 
                                           id="<?php echo esc_attr($field_id); ?>" 
                                           name="<?php echo esc_attr($field_id); ?>" 
                                           value="1" 
                                           <?php checked($field_value, true); ?>
                                    >
                                    <?php
                                    break;
                                    
                                case 'select':
                                    ?>
                                    <select id="<?php echo esc_attr($field_id); ?>" 
                                            name="<?php echo esc_attr($field_id); ?>">
                                        <?php foreach ($field['options'] as $option_value => $option_label): ?>
                                            <option value="<?php echo esc_attr($option_value); ?>" 
                                                    <?php selected($field_value, $option_value); ?>>
                                                <?php echo esc_html($option_label); ?>
                                            </option>
                                        <?php endforeach; ?>
                                    </select>
                                    <?php
                                    break;
                                    
                                default:
                                    ?>
                                    <input type="text" 
                                           id="<?php echo esc_attr($field_id); ?>" 
                                           name="<?php echo esc_attr($field_id); ?>" 
                                           value="<?php echo esc_attr($field_value); ?>" 
                                           class="regular-text">
                                    <?php
                            }
                            
                            // Display field description if provided
                            if (isset($field['description'])) {
                                echo '<p class="description">' . esc_html($field['description']) . '</p>';
                            }
                            ?>
                        </td>
                    </tr>
                <?php endforeach; ?>
            </tbody>
        </table>
        
        <p class="submit">
            <input type="submit" name="submit" id="submit" class="button button-primary" value="Save Changes">
        </p>
    </form>
    <?php
}

// Set the content callback
$content_callback = 'render_options_content';

// Include the header template
include_once plugin_dir_path(__FILE__) . 'custom-header.php';
?> 