<?php

/**
 * The admin-specific functionality of the plugin.
 *
 * @since      1.0.0
 */
class Custom_Plugin_Template_Admin {

    /**
     * The ID of this plugin.
     *
     * @since    1.0.0
     * @access   private
     * @var      string    $plugin_name    The ID of this plugin.
     */
    private $plugin_name;

    /**
     * The version of this plugin.
     *
     * @since    1.0.0
     * @access   private
     * @var      string    $version    The current version of this plugin.
     */
    private $version;

   
    /**
     * Initialize the class and set its properties.
     *
     * @since    1.0.0
     * @param      string    $plugin_name       The name of this plugin.
     * @param      string    $version    The version of this plugin.
     */
    public function __construct( $plugin_name, $version ) {

        $this->plugin_name = $plugin_name;
        $this->version = $version;
    }
    /**
     * Register the stylesheets for the admin area.
     *
     * @since    1.0.0
     */
    public function enqueue_styles() {
        wp_enqueue_style( $this->plugin_name, plugin_dir_url( __FILE__ ) . 'css/custom-admin.css', array(), $this->version, 'all' );
        
        // Add inline CSS to hide the default admin title
        $current_screen = get_current_screen();
        if (isset($current_screen->id) && 
            (strpos($current_screen->id, 'custom-plugin-template') !== false || 
             strpos($current_screen->id, 'toplevel_page_custom-plugin-template') !== false)) {
            
            wp_add_inline_style($this->plugin_name, '
                body.toplevel_page_custom-plugin-template h1.wp-heading-inline:not(.custom-intro h1),
                body.toplevel_page_custom-plugin-template .wrap > h1:not(.custom-intro h1),
                body.custom-plugin-template_page_custom-settings h1.wp-heading-inline:not(.custom-intro h1),
                body.custom-plugin-template_page_custom-settings .wrap > h1:not(.custom-intro h1) {
                    display: none !important;
                }
                
                .custom-intro h1 {
                    display: block !important;
                }
            ');
        }
    }

    /**
     * Add an options page under the Settings submenu
     *
     * @since  1.0.0
     */
    public function add_plugin_admin_menu() {
        
        add_menu_page(
            __( 'Service Charge Manager', 'custom-plugin-template' ),
            __( 'Service Charge', 'custom-plugin-template' ),
            'manage_options',
            $this->plugin_name,
            array( $this, 'display_plugin_page' ),
            'dashicons-money-alt',
            100
        );

        // Add submenu pages
        add_submenu_page(
            $this->plugin_name,
            __( 'Dashboard', 'custom-plugin-template' ),
            __( 'Dashboard', 'custom-plugin-template' ),
            'manage_options',
            $this->plugin_name,
            array( $this, 'display_plugin_page' )
        );
        
        // Add Service Charge Settings submenu
        add_submenu_page(
            $this->plugin_name,
            __( 'Service Charge Settings', 'custom-plugin-template' ),
            __( 'Settings', 'custom-plugin-template' ),
            'manage_options',
            $this->plugin_name . '-settings',
            array( $this, 'display_settings_page' )
        );
    }

    /**
     * Render the main plugin page.
     *
     * @since    1.0.0
     */
    public function display_plugin_page() {
        include_once( 'partials/test-page.php' );
    }
    
    /**
     * Render the settings page.
     *
     * @since    1.0.0
     */
    public function display_settings_page() {
        include_once( 'partials/service-charge-settings.php' );
    }
    
    /**
     * Get the service charge percentage.
     *
     * @since    1.0.0
     * @return   int    The service charge percentage.
     */
    public function get_service_charge_percentage() {
        $options = get_option('service_charge_options', array());
        $percentage = isset($options['service_charge_percentage']) ? intval($options['service_charge_percentage']) : 10;
        
        // Ensure percentage is within valid range
        $percentage = max(1, min(100, $percentage));
        
        return $percentage;
    }
    
    /**
     * Check if service charge is enabled.
     *
     * @since    1.0.0
     * @return   bool    Whether service charge is enabled.
     */
    public function is_service_charge_enabled() {
        $options = get_option('service_charge_options', array());
        return isset($options['enable_service_charge']) ? (bool) $options['enable_service_charge'] : true;
    }
    
    /**
     * Get the service charge description.
     *
     * @since    1.0.0
     * @return   string    The service charge description.
     */
    public function get_service_charge_description() {
        $options = get_option('service_charge_options', array());
        $description = isset($options['service_charge_description']) ? $options['service_charge_description'] : 'A service charge of {percentage}% will be added to your bill.';
        
        // Replace placeholder with actual percentage
        $percentage = $this->get_service_charge_percentage();
        $description = str_replace('{percentage}', $percentage, $description);
        
        return $description;
    }
} 